module wind;

import <cstdarg>;
import <cstdint>;
import <array>;
import <memory>;
import allegro;
import allegro.font_addon;
import :base;
import :hex;
import :console.base;
import :console.font;
import :string;
import :bitmap;
import :file;
import :color;
import :array;

namespace WIND
{
	namespace CONSOLE
	{
		inline constexpr int32_t FONT_GLYPH_SPAN = 16;
	}
}

namespace wind
{
	namespace console
	{
		struct font_data_tag_t
		{
			size_t m_start{ 0 };
			size_t m_count{ 0 };
			ALLEGRO::BITMAP m_bitmap{};
			array_t<ALLEGRO::BITMAP> m_glyphs{};
		};

		using glyph_data_t = std::array<uint8_t, WIND::CONSOLE::FONT_GLYPH_SIZE>;

		const std::array<glyph_data_t, (1 << 8)> default_font_data 
		{ {
			{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
			{ 0x7e, 0x81, 0xa5, 0x81, 0xbd, 0x99, 0x81, 0x7e},
			{ 0x7e, 0xff, 0xdb, 0xff, 0xc3, 0xe7, 0xff, 0x7e},
			{ 0x6c, 0xfe, 0xfe, 0xfe, 0x7c, 0x38, 0x10, 0x00},
			{ 0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x38, 0x10, 0x00},
			{ 0x38, 0x7c, 0x38, 0xfe, 0xfe, 0xd6, 0x10, 0x38},
			{ 0x10, 0x10, 0x38, 0x7c, 0xfe, 0x7c, 0x10, 0x38},
			{ 0x00, 0x00, 0x18, 0x3c, 0x3c, 0x18, 0x00, 0x00},
			{ 0xff, 0xff, 0xe7, 0xc3, 0xc3, 0xe7, 0xff, 0xff},
			{ 0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c, 0x00},
			{ 0xff, 0xc3, 0x99, 0xbd, 0xbd, 0x99, 0xc3, 0xff},
			{ 0x0f, 0x07, 0x0f, 0x7d, 0xcc, 0xcc, 0xcc, 0x78},
			{ 0x3c, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18},
			{ 0x3f, 0x33, 0x3f, 0x30, 0x30, 0x70, 0xf0, 0xe0},
			{ 0x7f, 0x63, 0x7f, 0x63, 0x63, 0x67, 0xe6, 0xc0},
			{ 0x99, 0x5a, 0x3c, 0xe7, 0xe7, 0x3c, 0x5a, 0x99},
			{ 0x80, 0xe0, 0xf8, 0xfe, 0xf8, 0xe0, 0x80, 0x00},
			{ 0x02, 0x0e, 0x3e, 0xfe, 0x3e, 0x0e, 0x02, 0x00},
			{ 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x7e, 0x3c, 0x18},
			{ 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x66, 0x00},
			{ 0x7f, 0xdb, 0xdb, 0x7b, 0x1b, 0x1b, 0x1b, 0x00},
			{ 0x7e, 0xc3, 0x78, 0xcc, 0xcc, 0x78, 0x8c, 0xf8},
			{ 0x00, 0x00, 0x00, 0x00, 0x7e, 0x7e, 0x7e, 0x00},
			{ 0x18, 0x3c, 0x7e, 0x18, 0x7e, 0x3c, 0x18, 0xff},
			{ 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x00},
			{ 0x18, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00},
			{ 0x00, 0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00},
			{ 0x00, 0x30, 0x60, 0xfe, 0x60, 0x30, 0x00, 0x00},
			{ 0x00, 0x00, 0xc0, 0xc0, 0xc0, 0xfe, 0x00, 0x00},
			{ 0x00, 0x24, 0x66, 0xff, 0x66, 0x24, 0x00, 0x00},
			{ 0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff, 0x00, 0x00},
			{ 0x00, 0xff, 0xff, 0x7e, 0x3c, 0x18, 0x00, 0x00},
			{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
			{ 0x30, 0x78, 0x78, 0x30, 0x30, 0x00, 0x30, 0x00},
			{ 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x00},
			{ 0x6c, 0x6c, 0xfe, 0x6c, 0xfe, 0x6c, 0x6c, 0x00},
			{ 0x30, 0x7c, 0xc0, 0x78, 0x0c, 0xf8, 0x30, 0x00},
			{ 0x00, 0xc6, 0xcc, 0x18, 0x30, 0x66, 0xc6, 0x00},
			{ 0x38, 0x6c, 0x38, 0x76, 0xdc, 0xcc, 0x76, 0x00},
			{ 0x60, 0x60, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00},
			{ 0x18, 0x30, 0x60, 0x60, 0x60, 0x30, 0x18, 0x00},
			{ 0x60, 0x30, 0x18, 0x18, 0x18, 0x30, 0x60, 0x00},
			{ 0x00, 0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00},
			{ 0x00, 0x30, 0x30, 0xfc, 0x30, 0x30, 0x00, 0x00},
			{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0x30, 0x60},
			{ 0x00, 0x00, 0x00, 0xfc, 0x00, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00},
			{ 0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0x80, 0x00},
			{ 0x78, 0xcc, 0xdc, 0xfc, 0xec, 0xcc, 0x78, 0x00},
			{ 0x30, 0xf0, 0x30, 0x30, 0x30, 0x30, 0xfc, 0x00},
			{ 0x78, 0xcc, 0x0c, 0x38, 0x60, 0xcc, 0xfc, 0x00},
			{ 0x78, 0xcc, 0x0c, 0x38, 0x0c, 0xcc, 0x78, 0x00},
			{ 0x1c, 0x3c, 0x6c, 0xcc, 0xfe, 0x0c, 0x0c, 0x00},
			{ 0xfc, 0xc0, 0xf8, 0x0c, 0x0c, 0xcc, 0x78, 0x00},
			{ 0x38, 0x60, 0xc0, 0xf8, 0xcc, 0xcc, 0x78, 0x00},
			{ 0xfc, 0xcc, 0x0c, 0x18, 0x30, 0x60, 0x60, 0x00},
			{ 0x78, 0xcc, 0xcc, 0x78, 0xcc, 0xcc, 0x78, 0x00},
			{ 0x78, 0xcc, 0xcc, 0x7c, 0x0c, 0x18, 0x70, 0x00},
			{ 0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x00},
			{ 0x00, 0x00, 0x30, 0x30, 0x00, 0x70, 0x30, 0x60},
			{ 0x18, 0x30, 0x60, 0xc0, 0x60, 0x30, 0x18, 0x00},
			{ 0x00, 0x00, 0xfc, 0x00, 0xfc, 0x00, 0x00, 0x00},
			{ 0x60, 0x30, 0x18, 0x0c, 0x18, 0x30, 0x60, 0x00},
			{ 0x78, 0xcc, 0x0c, 0x18, 0x30, 0x00, 0x30, 0x00},
			{ 0x7c, 0xc6, 0xde, 0xde, 0xde, 0xc0, 0x78, 0x00},
			{ 0x30, 0x78, 0xcc, 0xcc, 0xfc, 0xcc, 0xcc, 0x00},
			{ 0xfc, 0x66, 0x66, 0x7c, 0x66, 0x66, 0xfc, 0x00},
			{ 0x3c, 0x66, 0xc0, 0xc0, 0xc0, 0x66, 0x3c, 0x00},
			{ 0xfc, 0x6c, 0x66, 0x66, 0x66, 0x6c, 0xfc, 0x00},
			{ 0xfe, 0x62, 0x68, 0x78, 0x68, 0x62, 0xfe, 0x00},
			{ 0xfe, 0x62, 0x68, 0x78, 0x68, 0x60, 0xf0, 0x00},
			{ 0x3c, 0x66, 0xc0, 0xc0, 0xce, 0x66, 0x3e, 0x00},
			{ 0xcc, 0xcc, 0xcc, 0xfc, 0xcc, 0xcc, 0xcc, 0x00},
			{ 0x78, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00},
			{ 0x1e, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0x78, 0x00},
			{ 0xe6, 0x66, 0x6c, 0x78, 0x6c, 0x66, 0xe6, 0x00},
			{ 0xf0, 0x60, 0x60, 0x60, 0x62, 0x66, 0xfe, 0x00},
			{ 0xc6, 0xee, 0xfe, 0xd6, 0xc6, 0xc6, 0xc6, 0x00},
			{ 0xc6, 0xe6, 0xf6, 0xde, 0xce, 0xc6, 0xc6, 0x00},
			{ 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0x6c, 0x38, 0x00},
			{ 0xfc, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00},
			{ 0x78, 0xcc, 0xcc, 0xcc, 0xdc, 0x78, 0x1c, 0x00},
			{ 0xfc, 0x66, 0x66, 0x7c, 0x78, 0x6c, 0xe6, 0x00},
			{ 0x78, 0xcc, 0xe0, 0x38, 0x1c, 0xcc, 0x78, 0x00},
			{ 0xfc, 0xb4, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00},
			{ 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xfc, 0x00},
			{ 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x00},
			{ 0xc6, 0xc6, 0xc6, 0xd6, 0xfe, 0xee, 0xc6, 0x00},
			{ 0xc6, 0xc6, 0x6c, 0x38, 0x6c, 0xc6, 0xc6, 0x00},
			{ 0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x30, 0x78, 0x00},
			{ 0xfe, 0xcc, 0x98, 0x30, 0x62, 0xc6, 0xfe, 0x00},
			{ 0x78, 0x60, 0x60, 0x60, 0x60, 0x60, 0x78, 0x00},
			{ 0xc0, 0x60, 0x30, 0x18, 0x0c, 0x06, 0x02, 0x00},
			{ 0x78, 0x18, 0x18, 0x18, 0x18, 0x18, 0x78, 0x00},
			{ 0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff},
			{ 0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x76, 0x00},
			{ 0xe0, 0x60, 0x7c, 0x66, 0x66, 0x66, 0xbc, 0x00},
			{ 0x00, 0x00, 0x78, 0xcc, 0xc0, 0xcc, 0x78, 0x00},
			{ 0x1c, 0x0c, 0x0c, 0x7c, 0xcc, 0xcc, 0x76, 0x00},
			{ 0x00, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00},
			{ 0x38, 0x6c, 0x60, 0xf0, 0x60, 0x60, 0xf0, 0x00},
			{ 0x00, 0x00, 0x76, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8},
			{ 0xe0, 0x60, 0x6c, 0x76, 0x66, 0x66, 0xe6, 0x00},
			{ 0x30, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00},
			{ 0x18, 0x00, 0x78, 0x18, 0x18, 0x18, 0xd8, 0x70},
			{ 0xe0, 0x60, 0x66, 0x6c, 0x78, 0x6c, 0xe6, 0x00},
			{ 0x70, 0x30, 0x30, 0x30, 0x30, 0x30, 0x78, 0x00},
			{ 0x00, 0x00, 0xec, 0xfe, 0xd6, 0xc6, 0xc6, 0x00},
			{ 0x00, 0x00, 0xf8, 0xcc, 0xcc, 0xcc, 0xcc, 0x00},
			{ 0x00, 0x00, 0x78, 0xcc, 0xcc, 0xcc, 0x78, 0x00},
			{ 0x00, 0x00, 0xdc, 0x66, 0x66, 0x7c, 0x60, 0xf0},
			{ 0x00, 0x00, 0x76, 0xcc, 0xcc, 0x7c, 0x0c, 0x1e},
			{ 0x00, 0x00, 0xd8, 0x6c, 0x6c, 0x60, 0xf0, 0x00},
			{ 0x00, 0x00, 0x7c, 0xc0, 0x78, 0x0c, 0xf8, 0x00},
			{ 0x10, 0x30, 0x7c, 0x30, 0x30, 0x34, 0x18, 0x00},
			{ 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 0x00},
			{ 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0x78, 0x30, 0x00},
			{ 0x00, 0x00, 0xc6, 0xc6, 0xd6, 0xfe, 0x6c, 0x00},
			{ 0x00, 0x00, 0xc6, 0x6c, 0x38, 0x6c, 0xc6, 0x00},
			{ 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xf8},
			{ 0x00, 0x00, 0xfc, 0x98, 0x30, 0x64, 0xfc, 0x00},
			{ 0x1c, 0x30, 0x30, 0xe0, 0x30, 0x30, 0x1c, 0x00},
			{ 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00},
			{ 0xe0, 0x30, 0x30, 0x1c, 0x30, 0x30, 0xe0, 0x00},
			{ 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
			{ 0x10, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0xfe, 0x00},
			{ 0x78, 0xcc, 0xc0, 0xcc, 0x78, 0x18, 0x0c, 0x78},
			{ 0x00, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00},
			{ 0x1c, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00},
			{ 0x7e, 0xc3, 0x3c, 0x06, 0x3e, 0x66, 0x3f, 0x00},
			{ 0xcc, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00},
			{ 0xe0, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00},
			{ 0x30, 0x30, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00},
			{ 0x00, 0x00, 0x7c, 0xc0, 0xc0, 0x7c, 0x06, 0x3c},
			{ 0x7e, 0xc3, 0x3c, 0x66, 0x7e, 0x60, 0x3c, 0x00},
			{ 0xcc, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00},
			{ 0xe0, 0x00, 0x78, 0xcc, 0xfc, 0xc0, 0x78, 0x00},
			{ 0xcc, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00},
			{ 0x7c, 0xc6, 0x38, 0x18, 0x18, 0x18, 0x3c, 0x00},
			{ 0xe0, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00},
			{ 0xcc, 0x30, 0x78, 0xcc, 0xcc, 0xfc, 0xcc, 0x00},
			{ 0x30, 0x30, 0x00, 0x78, 0xcc, 0xfc, 0xcc, 0x00},
			{ 0x1c, 0x00, 0xfc, 0x60, 0x78, 0x60, 0xfc, 0x00},
			{ 0x00, 0x00, 0x7f, 0x0c, 0x7f, 0xcc, 0x7f, 0x00},
			{ 0x3e, 0x6c, 0xcc, 0xfe, 0xcc, 0xcc, 0xce, 0x00},
			{ 0x78, 0xcc, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00},
			{ 0x00, 0xcc, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00},
			{ 0x00, 0xe0, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00},
			{ 0x78, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00},
			{ 0x00, 0xe0, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00},
			{ 0x00, 0xcc, 0x00, 0xcc, 0xcc, 0xfc, 0x0c, 0xf8},
			{ 0xc6, 0x38, 0x7c, 0xc6, 0xc6, 0x7c, 0x38, 0x00},
			{ 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0x78, 0x00},
			{ 0x18, 0x18, 0x7e, 0xc0, 0xc0, 0x7e, 0x18, 0x18},
			{ 0x38, 0x6c, 0x64, 0xf0, 0x60, 0xe6, 0xfc, 0x00},
			{ 0xcc, 0xcc, 0x78, 0xfc, 0x30, 0xfc, 0x30, 0x00},
			{ 0xf0, 0xd8, 0xd8, 0xf4, 0xcc, 0xde, 0xcc, 0x0e},
			{ 0x0e, 0x1b, 0x18, 0x7e, 0x18, 0x18, 0xd8, 0x70},
			{ 0x1c, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0x7e, 0x00},
			{ 0x38, 0x00, 0x70, 0x30, 0x30, 0x30, 0x78, 0x00},
			{ 0x00, 0x1c, 0x00, 0x78, 0xcc, 0xcc, 0x78, 0x00},
			{ 0x00, 0x1c, 0x00, 0xcc, 0xcc, 0xcc, 0x7e, 0x00},
			{ 0x00, 0xf8, 0x00, 0xf8, 0xcc, 0xcc, 0xcc, 0x00},
			{ 0xfc, 0x00, 0xcc, 0xec, 0xfc, 0xdc, 0xcc, 0x00},
			{ 0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00, 0x00},
			{ 0x3c, 0x66, 0x66, 0x3c, 0x00, 0x7e, 0x00, 0x00},
			{ 0x30, 0x00, 0x30, 0x60, 0xc0, 0xcc, 0x78, 0x00},
			{ 0x00, 0x00, 0x00, 0xfc, 0xc0, 0xc0, 0x00, 0x00},
			{ 0x00, 0x00, 0x00, 0xfc, 0x0c, 0x0c, 0x00, 0x00},
			{ 0xc6, 0xcc, 0xd8, 0x3e, 0x63, 0xce, 0x98, 0x1f},
			{ 0xc6, 0xcc, 0xd8, 0xf3, 0x67, 0xcf, 0x9f, 0x03},
			{ 0x00, 0x18, 0x00, 0x18, 0x18, 0x3c, 0x3c, 0x18},
			{ 0x00, 0x33, 0x66, 0xcc, 0x66, 0x33, 0x00, 0x00},
			{ 0x00, 0xcc, 0x66, 0x33, 0x66, 0xcc, 0x00, 0x00},
			{ 0x22, 0x88, 0x22, 0x88, 0x22, 0x88, 0x22, 0x88},
			{ 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa},
			{ 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77},
			{ 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18},
			{ 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0x18, 0x18},
			{ 0x18, 0x18, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18},
			{ 0x36, 0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36},
			{ 0x00, 0x00, 0x00, 0x00, 0xfe, 0x36, 0x36, 0x36},
			{ 0x00, 0x00, 0xf8, 0x18, 0xf8, 0x18, 0x18, 0x18},
			{ 0x36, 0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36, 0x36},
			{ 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36},
			{ 0x00, 0x00, 0xfe, 0x06, 0xf6, 0x36, 0x36, 0x36},
			{ 0x36, 0x36, 0xf6, 0x06, 0xfe, 0x00, 0x00, 0x00},
			{ 0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00, 0x00},
			{ 0x18, 0x18, 0xf8, 0x18, 0xf8, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0x18, 0x18},
			{ 0x18, 0x18, 0x18, 0x18, 0x1f, 0x00, 0x00, 0x00},
			{ 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18},
			{ 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x18, 0x18},
			{ 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00},
			{ 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0x18, 0x18},
			{ 0x18, 0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18},
			{ 0x36, 0x36, 0x36, 0x36, 0x37, 0x36, 0x36, 0x36},
			{ 0x36, 0x36, 0x37, 0x30, 0x3f, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36, 0x36},
			{ 0x36, 0x36, 0xf7, 0x00, 0xff, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0xff, 0x00, 0xf7, 0x36, 0x36, 0x36},
			{ 0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36},
			{ 0x00, 0x00, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00},
			{ 0x36, 0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36},
			{ 0x18, 0x18, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00},
			{ 0x36, 0x36, 0x36, 0x36, 0xff, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0xff, 0x00, 0xff, 0x18, 0x18, 0x18},
			{ 0x00, 0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36},
			{ 0x36, 0x36, 0x36, 0x36, 0x3f, 0x00, 0x00, 0x00},
			{ 0x18, 0x18, 0x1f, 0x18, 0x1f, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18},
			{ 0x00, 0x00, 0x00, 0x00, 0x3f, 0x36, 0x36, 0x36},
			{ 0x36, 0x36, 0x36, 0x36, 0xf7, 0x36, 0x36, 0x36},
			{ 0x18, 0x18, 0xff, 0x00, 0xff, 0x18, 0x18, 0x18},
			{ 0x18, 0x18, 0x18, 0x18, 0xf8, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18},
			{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff},
			{ 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff},
			{ 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0},
			{ 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f},
			{ 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0x76, 0xdc, 0xc8, 0xdc, 0x76, 0x00},
			{ 0x00, 0x78, 0xcc, 0xf8, 0xcc, 0xf8, 0xc0, 0xc0},
			{ 0x00, 0xfe, 0xc6, 0xc0, 0xc0, 0xc0, 0xc0, 0x00},
			{ 0x00, 0xfe, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00},
			{ 0xfe, 0x66, 0x30, 0x18, 0x30, 0x66, 0xfe, 0x00},
			{ 0x00, 0x00, 0x7e, 0xcc, 0xcc, 0xcc, 0x78, 0x00},
			{ 0x00, 0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0xc0},
			{ 0x00, 0x76, 0xdc, 0x18, 0x18, 0x18, 0x18, 0x00},
			{ 0xfc, 0x30, 0x78, 0xcc, 0xcc, 0x78, 0x30, 0xfc},
			{ 0x38, 0x6c, 0xc6, 0xfe, 0xc6, 0x6c, 0x38, 0x00},
			{ 0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x6c, 0xee, 0x00},
			{ 0x1c, 0x30, 0x18, 0x7c, 0xcc, 0xcc, 0x78, 0x00},
			{ 0x00, 0x00, 0x7e, 0xdb, 0xdb, 0x7e, 0x00, 0x00},
			{ 0x06, 0x0c, 0x7e, 0xdb, 0xdb, 0x7e, 0x60, 0xc0},
			{ 0x3c, 0x60, 0xc0, 0xfc, 0xc0, 0x60, 0x3c, 0x00},
			{ 0x78, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x00},
			{ 0x00, 0xfc, 0x00, 0xfc, 0x00, 0xfc, 0x00, 0x00},
			{ 0x30, 0x30, 0xfc, 0x30, 0x30, 0x00, 0xfc, 0x00},
			{ 0x60, 0x30, 0x18, 0x30, 0x60, 0x00, 0xfc, 0x00},
			{ 0x18, 0x30, 0x60, 0x30, 0x18, 0x00, 0xfc, 0x00},
			{ 0x0e, 0x1b, 0x1b, 0x18, 0x18, 0x18, 0x18, 0x18},
			{ 0x18, 0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8, 0x70},
			{ 0x30, 0x30, 0x00, 0xfc, 0x00, 0x30, 0x30, 0x00},
			{ 0x00, 0x72, 0x9c, 0x00, 0x72, 0x9c, 0x00, 0x00},
			{ 0x38, 0x6c, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00},
			{ 0x0f, 0x0c, 0x0c, 0x0c, 0xec, 0x6c, 0x3c, 0x1c},
			{ 0x78, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00},
			{ 0x78, 0x0c, 0x38, 0x60, 0x7c, 0x00, 0x00, 0x00},
			{ 0x00, 0x00, 0x3c, 0x3c, 0x3c, 0x3c, 0x00, 0x00},
			{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }
		} };

		auto convert_font_glyph_from_data(const uint8_t* data, ALLEGRO::BITMAP& glyph) -> bool;
		auto convert_font_glyph_to_data(const ALLEGRO::BITMAP& glyph, uint8_t* data) -> bool;

		auto create_font(font_t& font, const uint8_t* data, size_t count, int32_t start) -> int32_t
		{
			size_t h = (count >> WIND::CONSOLE::FONT_GLYPH_SHIFT);
			size_t w = WIND::CONSOLE::FONT_GLYPH_SIZE;
			size_t rows = (count / WIND::CONSOLE::FONT_GLYPH_SPAN) + ((count % WIND::CONSOLE::FONT_GLYPH_SPAN) ? 1 : 0);
			ALLEGRO::RECTANGLE<size_t> rect{ 0, 0, WIND::CONSOLE::FONT_GLYPH_SIZE, WIND::CONSOLE::FONT_GLYPH_SIZE };
			ALLEGRO::SIZE<size_t> bsize{ WIND::CONSOLE::FONT_GLYPH_SPAN << WIND::CONSOLE::FONT_GLYPH_SHIFT, rows << WIND::CONSOLE::FONT_GLYPH_SHIFT };

			font->m_count = 0;
			font->m_start = 0;

			font->m_bitmap = al::create_bitmap(bsize);
			if (!font->m_bitmap)
			{
				return -1;
			}

			font->m_glyphs = array_t<ALLEGRO::BITMAP>(count);

			size_t index = 0;

			for (size_t j = 0; j < h; ++j)
			{
				rect.position.y = (index / WIND::CONSOLE::FONT_GLYPH_SPAN) << WIND::CONSOLE::FONT_GLYPH_SHIFT;

				for (size_t i = 0; i < w; ++i)
				{
					rect.position.x = (index % WIND::CONSOLE::FONT_GLYPH_SPAN) << WIND::CONSOLE::FONT_GLYPH_SHIFT;

					ALLEGRO::BITMAP glyph = al::create_sub_bitmap(font->m_bitmap, rect);

					if (!glyph)
					{
						return -1;
					}

					convert_font_glyph_from_data(data + (index * WIND::CONSOLE::FONT_GLYPH_SIZE), glyph);

					font->m_glyphs[index] = glyph;

					++index;
				}
			}

			font->m_count = count;
			font->m_start = start;

			return 0;
		}

		auto create_font_from_array(const uint8_t* data, size_t count, int32_t start) -> font_t;

		auto create_font_default() -> font_t
		{
			constexpr size_t size = (sizeof(default_font_data) / sizeof(uint8_t)) >> WIND::CONSOLE::FONT_GLYPH_SHIFT;

			return create_font_from_array((uint8_t*)default_font_data.data(), size, 0);
		}

		auto create_font_from_array(const uint8_t* data, size_t count, int32_t start) -> font_t
		{
			font_t font = std::make_shared<font_data_t>();

			if (font)
			{
				if (create_font(font, data, count, start) < 0)
				{
					font.reset();
				}
			}

			return font;
		}

		auto save_font(const font_t& font, const wind::string_t& filename) -> bool
		{
			ALLEGRO::ASSERT(font);

			bool rv = false;

			ALLEGRO::FILE file = al::fopen(filename.c_str(), "rb");

			if (file)
			{
				rv = save_font_f(file, font);
				file.reset();
			}

			return rv;
		}

		auto save_font_f(ALLEGRO::FILE& file, const font_t& font) -> bool
		{
			ALLEGRO::ASSERT(file && font);

			bool rv = false;
			string_t out;

			file << "{\n\t\"font\" :\n\t{\n";
			file << "\t\t\"start\": ";
			al::fprintf(file, "%lu", font->m_start);
			file << ",\n";
			file << "\t\t\"count\": ";
			al::fprintf(file, "%lu", font->m_count);
			file << ",\n";
			file << "\t\t\"data\":\n\t\t[\n";

			for (int32_t index = 0; index < font->m_count; ++index)
			{
				uint8_t data[WIND::CONSOLE::FONT_GLYPH_SIZE] =
				{
					0, 0, 0, 0, 0, 0, 0, 0
				};

				out.clear();

				convert_font_glyph_to_data(font->m_glyphs[index], data);

				for (int32_t j = 0; j < WIND::CONSOLE::FONT_GLYPH_SIZE; ++j)
				{
					out.append("0x");

					out.push_back(hex_to_char((data[j] & 0xf0) >> 4));
					out.push_back(hex_to_char(data[j] & 0xf));

					if (j < (WIND::CONSOLE::FONT_GLYPH_SIZE - 1))
					{
						out.append(", ");
					}
				}
				file << "\t\t\t{ " << out << " }";

				if (index < (font->m_count - 1))
				{
					file << ',';
				}

				out.clear();
				out.append(" // 0x");
				out.push_back(hex_to_char((index & 0xf0) >> 4));
				out.push_back(hex_to_char(index & 0xf));
				file << out;

				if ((index & 0xf) == 0xf)
				{
					file << '\n';
				}

				file << '\n';
			}

			file << "\t\t]\n\t}\n}\n\n";

			return true;
		}

		auto get_font_bitmap(font_t& font) -> ALLEGRO::BITMAP&
		{
			ALLEGRO::ASSERT(font);
			return font->m_bitmap;
		}

		auto get_font_bitmap(const font_t& font) -> const ALLEGRO::BITMAP&
		{
			ALLEGRO::ASSERT(font);
			return font->m_bitmap;
		}

		auto get_font_glyph(font_t& font, size_t index) -> ALLEGRO::BITMAP&
		{
			ALLEGRO::ASSERT(font);
			ALLEGRO::ASSERT(index >= font->m_start && index <= (font->m_start + font->m_count));

			return font->m_glyphs[index - font->m_start];
		}

		auto get_font_glyph(const font_t& font, size_t index) -> const ALLEGRO::BITMAP&
		{
			ALLEGRO::ASSERT(font);
			ALLEGRO::ASSERT(index >= font->m_start && index <= (font->m_start + font->m_count));

			return font->m_glyphs[index - font->m_start];
		}

		auto set_font_glyph(font_t& font, size_t index, const uint8_t* data) -> bool
		{
			ALLEGRO::ASSERT(font);
			ALLEGRO::ASSERT(index >= font->m_start && index <= (font->m_start + font->m_count));

			return convert_font_glyph_from_data(data, font->m_glyphs[index - font->m_start]);
		}

		auto set_font_glyph(font_t& font, size_t index, const wind::string_t& string) -> bool
		{
			ALLEGRO::ASSERT(font);
			ALLEGRO::ASSERT(index >= font->m_start && index <= (font->m_start + font->m_count));

			uint8_t data[WIND::CONSOLE::FONT_GLYPH_SIZE] = { 0, 0, 0, 0, 0, 0, 0, 0 };

			for (size_t i = 0; (i < string.length() && i < WIND::CONSOLE::FONT_GLYPH_SIZE); ++i)
			{
				data[i] = hex_from_char(string[i]);
			}

			return set_font_glyph(font, index, data);
		}

		auto draw_font_glyph(const font_t& font, ALLEGRO::COLOR color, const ALLEGRO::POINT<int32_t>& point, uint8_t index) -> void
		{
			ALLEGRO::ASSERT(font);
			ALLEGRO::ASSERT(index >= font->m_start && index <= (font->m_start + font->m_count));

			al::draw_tinted_bitmap(font->m_glyphs[index - font->m_start], color, point, 0);
		}

		auto draw_font(const font_t& font, ALLEGRO::COLOR color, const ALLEGRO::POINT<int32_t>& point, int32_t alignment, const wind::string_t& text) -> void
		{
			ALLEGRO::ASSERT(font);
			float w = text.length() << WIND::CONSOLE::FONT_GLYPH_SHIFT;
			ALLEGRO::POINT<int32_t> pos{ point };

			if (alignment == WIND::CONSOLE::FONT_ALIGNMENT_RIGHT)
			{
				pos.x -= w;
			}

			if (alignment == WIND::CONSOLE::FONT_ALIGNMENT_CENTRE)
			{
				pos.x -= (w * 0.5f);
			}

			for (auto it = text.cbegin(); it != text.cend(); ++it)
			{
				draw_font_glyph(font, color, pos, (*it).get_codepoint());
				pos.x += WIND::CONSOLE::FONT_GLYPH_SIZE;
			}
		}

		auto draw_font(const font_t& font, ALLEGRO::COLOR color, const ALLEGRO::POINT<int32_t>& point, int32_t alignment, const char* format, ...) -> void
		{
			va_list args;
			int32_t len;
			char* buffer;

			va_start(args, format);
			len = _vscprintf(format, args) // _vscprintf doesn't count
				+ 1; // terminating '\0'
			buffer = (char*)malloc(len * sizeof(char));

			if (NULL != buffer)
			{
				vsprintf_s(buffer, len, format, args);
				draw_font(font, color, point, alignment, wind::string_t(buffer));
				free(buffer);
			}

			va_end(args);
		}

		auto convert_font_glyph_from_data(const uint8_t* data, ALLEGRO::BITMAP& glyph) -> bool
		{
			bool rv = false;

			if (glyph)
			{
				ALLEGRO::POINT<int32_t> point{ 0,0 };
				ALLEGRO::BITMAP target{ al::get_target_bitmap() };
				uint8_t p{ 0 };
				float color{ 0.0f };

				al::set_target_bitmap(glyph);
				ALLEGRO::BITMAP_LOCKED_REGION region = al::lock_bitmap(glyph, ALLEGRO::PIXEL_FORMAT_ANY, ALLEGRO::BITMAP_LOCK_WRITE_ONLY);

				al::clear_to_color(wind::map_rgba_i(0xffffffff));

				if (region)
				{
					for (int32_t j = 0; j < WIND::CONSOLE::FONT_GLYPH_SIZE; ++j)
					{
						uint8_t p = data[j];

						for (int32_t i = 0; i < WIND::CONSOLE::FONT_GLYPH_SIZE; ++i)
						{
							float c = 0.0f;

							if (p & 0x1)
							{
								c = 1.0f;
							}

							point = { (WIND::CONSOLE::FONT_GLYPH_SIZE - 1) - i, j };

							al::put_pixel(point, al::map_rgba_f(c, c, c, c));
							p = p >> 1;
						}
					}

					al::unlock_bitmap(glyph);
					rv = true;
				}

				al::set_target_bitmap(target);
			}

			return rv;
		}

		auto convert_font_glyph_to_data(const ALLEGRO::BITMAP& glyph, uint8_t* data) -> bool
		{
			ALLEGRO::ASSERT((bool)glyph);

			al::lock_bitmap(glyph, ALLEGRO::PIXEL_FORMAT_ANY, ALLEGRO::BITMAP_LOCK_READ_ONLY);

			int32_t white = 0xffffffff;
			ALLEGRO::POINT<int32_t> point{ 0,0 };

			for (point.y = 0; point.y < WIND::CONSOLE::FONT_GLYPH_SIZE; ++point.y)
			{
				data[point.y] = 0;

				for (point.x = 0; point.x < WIND::CONSOLE::FONT_GLYPH_SIZE; ++point.x)
				{
					int32_t color = wind::unmap_rgb_i(al::get_pixel(glyph, point));

					data[point.y] <<= 1;

					if (white == color)
					{
						data[point.y] |= 0x1;
					}
				}
			}

			al::unlock_bitmap(glyph);
			return true;
		}
	}
}