module qrcodegen;

/*
 * QR Code generator library (C++)
 *
 * Copyright (c) Project Nayuki. (MIT License)
 * https://www.nayuki.io/page/qr-code-generator-library
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sub license, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * - The above copyright notice and this permission notice shall be included in
 *   all copies or substantial portions of the Software.
 * - The Software is provided "as is", without warranty of any kind, express or
 *   implied, including but not limited to the warranties of merchantability,
 *   fitness for a particular purpose and non infringement. In no event shall the
 *   authors or copyright holders be liable for any claim, damages or other
 *   liability, whether in an action of contract, tort or otherwise, arising from,
 *   out of or in connection with the Software or the use or other dealings in the
 *   Software.
 */

import <string>;
import std;
import <cstdint>;
import <cassert>;
import <sstream>;
import allegro;
import wind;

namespace qrcodegen
{
	/*
	 * A segment of character/binary/control data in a QR Code symbol.
	 * Instances of this class are immutable.
	 * The mid-level way to create a segment is to take the payload data
	 * and call a static factory function such as QrSegment::makeNumeric().
	 * The low-level way to create a segment is to custom-make the bit buffer
	 * and call the QrSegment() constructor with appropriate values.
	 * This segment class imposes no length restrictions, but QR Codes have restrictions.
	 * Even in the most favorable conditions, a QR Code can only hold 7089 characters of data.
	 * Any segment longer than this is meaningless for the purpose of generating QR Codes.
	 */
	class QrSegment final
	{
	public:
		/*
		 * Describes how a segment's data bits are interpreted. Immutable.
		 */
		class Mode final
		{
		public:
			static const Mode NUMERIC;
			static const Mode ALPHANUMERIC;
			static const Mode BYTE;
			static const Mode KANJI;
			static const Mode ECI;

			/*
			* (Package-private) Returns the mode indicator bits, which is an unsigned 4-bit value (range 0 to 15).
			*/
			int32_t getModeBits() const;

			/*
			* (Package-private) Returns the bit width of the character count field for a segment in
			* this mode in a QR Code at the given version number. The result is in the range [0, 16].
			*/
			int32_t numCharCountBits(int32_t ver) const;

		private:
			// The mode indicator bits, which is a uint4 value (range 0 to 15).
			int32_t m_modeBits{ 0 };

			// Number of character count bits for three different version ranges.
			std::array<int32_t, 3> m_numBitsCharCount{ 0, 0, 0 };

			/*-- Constructor --*/
			Mode(int32_t mode, int32_t cc0, int32_t cc1, int32_t cc2);
		};

		/*---- Static factory functions (mid level) ----*/

		/*
		* Returns a segment representing the given binary data encoded in
		* byte mode. All input byte vectors are acceptable. Any text string
		* can be converted to UTF-8 bytes and encoded as a byte mode segment.
		*/
		static QrSegment makeBytes(const std::vector<std::uint8_t>& data);

		/*
		* Returns a segment representing the given string of decimal digits encoded in numeric mode.
		*/
		static QrSegment makeNumeric(const char* digits);

		/*
		* Returns a segment representing the given text string encoded in alphanumeric mode.
		* The characters allowed are: 0 to 9, A to Z (uppercase only), space,
		* dollar, percent, asterisk, plus, hyphen, period, slash, colon.
		*/
		static QrSegment makeAlphanumeric(const char* text);

		/*
		* Returns a list of zero or more segments to represent the given text string. The result
		* may use various segment modes and switch modes to optimize the length of the bit stream.
		*/
		static std::vector<QrSegment> makeSegments(const char* text);

		/*
		* Returns a segment representing an Extended Channel Interpretation
		* (ECI) designator with the given assignment value.
		*/
		static QrSegment makeEci(long assignVal);

		/*---- Public static helper functions ----*/

		/*
		* Tests whether the given string can be encoded as a segment in numeric mode.
		* A string is encodable iff each character is in the range 0 to 9.
		*/
		static bool isNumeric(const char* text);

		/*
		* Tests whether the given string can be encoded as a segment in alphanumeric mode.
		* A string is encodable iff each character is in the following set: 0 to 9, A to Z
		* (uppercase only), space, dollar, percent, asterisk, plus, hyphen, period, slash, colon.
		*/
		static bool isAlphanumeric(const char* text);

		/*
		* Creates a new QR Code segment with the given attributes and data.
		* The character count (numCh) must agree with the mode and the bit buffer length,
		* but the constraint isn't checked. The given bit buffer is copied and stored.
		*/
		QrSegment(const Mode& md, int32_t numCh, const std::vector<bool>& dt);

		/*
		* Creates a new QR Code segment with the given parameters and data.
		* The character count (numCh) must agree with the mode and the bit buffer length,
		* but the constraint isn't checked. The given bit buffer is moved and stored.
		*/
		QrSegment(const Mode& md, int32_t numCh, std::vector<bool>&& dt);

		/*
		* Returns the mode field of this segment.
		*/
		const Mode& getMode() const;

		/*
		* Returns the character count field of this segment.
		*/
		int32_t getNumChars() const;

		/*
		* Returns the data bits of this segment.
		*/
		const std::vector<bool>& getData() const;

		// (Package-private) Calculates the number of bits needed to encode the given segments at
		// the given version. Returns a non-negative number if successful. Otherwise returns -1 if a
		// segment has too many characters to fit its length field, or the total bits exceeds INT_MAX.
		static int32_t getTotalBits(const std::vector<QrSegment>& segs, int32_t version);

	private:
		/* The mode indicator of this segment. Accessed through getMode(). */
		const Mode* m_mode{ nullptr };

		/* The length of this segment's unencoded data. Measured in characters for
		* numeric/alphanumeric/kanji mode, bytes for byte mode, and 0 for ECI mode.
		* Always zero or positive. Not the same as the data's bit length.
		* Accessed through getNumChars(). */
		int32_t m_numChars{ 0 };

		/* The data bits of this segment. Accessed through getData(). */
		std::vector<bool> m_data{};

		/* The set of all legal characters in alphanumeric mode, where
		 * each character value maps to the index in the string. */
		static const char* ALPHANUMERIC_CHARSET;
	};

	/*
	 * A QR Code symbol, which is a type of two-dimension barcode.
	 * Invented by Denso Wave and described in the ISO/IEC 18004 standard.
	 * Instances of this class represent an immutable square grid of dark and light cells.
	 * The class provides static factory functions to create a QR Code from text or binary data.
	 * The class covers the QR Code Model 2 specification, supporting all versions (sizes)
	 * from 1 to 40, all 4 error correction levels, and 4 character encoding modes.
	 *
	 * Ways to create a QR Code object:
	 * - High level: Take the payload data and call QrCode::encodeText() or QrCode::encodeBinary().
	 * - Mid level: Custom-make the list of segments and call QrCode::encodeSegments().
	 * - Low level: Custom-make the array of data codeword bytes (including
	 *   segment headers and final padding, excluding error correction codewords),
	 *   supply the appropriate version number, and call the QrCode() constructor.
	 * (Note that all ways require supplying the desired error correction level.)
	 */
	class QrCode final
	{
	public:
		/*
		 * The error correction level in a QR Code symbol.
		 */
		enum class Ecc
		{
			LOW = 0,  // The QR Code can tolerate about  7% erroneous codewords
			MEDIUM,  // The QR Code can tolerate about 15% erroneous codewords
			QUARTILE,  // The QR Code can tolerate about 25% erroneous codewords
			HIGH,  // The QR Code can tolerate about 30% erroneous codewords
		};

		/*
		 * Returns a QR Code representing the given Unicode text string at the given error correction level.
		 * As a conservative upper bound, this function is guaranteed to succeed for strings that have 2953 or fewer
		 * UTF-8 code units (not Unicode code points) if the low error correction level is used. The smallest possible
		 * QR Code version is automatically chosen for the output. The ECC level of the result may be higher than
		 * the ecl argument if it can be done without increasing the version.
		 */
		static QrCode encodeText(const char* text, Ecc ecl);

		/*
		 * Returns a QR Code representing the given binary data at the given error correction level.
		 * This function always encodes using the binary segment mode, not any text mode. The maximum number of
		 * bytes allowed is 2953. The smallest possible QR Code version is automatically chosen for the output.
		 * The ECC level of the result may be higher than the ecl argument if it can be done without increasing the version.
		 */
		static QrCode encodeBinary(const std::vector<std::uint8_t>& data, Ecc ecl);

		/*---- Static factory functions (mid level) ----*/

		/*
		 * Returns a QR Code representing the given segments with the given encoding parameters.
		 * The smallest possible QR Code version within the given range is automatically
		 * chosen for the output. Iff boostEcl is true, then the ECC level of the result
		 * may be higher than the ecl argument if it can be done without increasing the
		 * version. The mask number is either between 0 to 7 (inclusive) to force that
		 * mask, or -1 to automatically choose an appropriate mask (which may be slow).
		 * This function allows the user to create a custom sequence of segments that switches
		 * between modes (such as alphanumeric and byte) to encode text in less space.
		 * This is a mid-level API; the high-level API is encodeText() and encodeBinary().
		 */
		static QrCode encodeSegments(const std::vector<QrSegment>& segs, Ecc ecl,
			int32_t minVersion = 1, int32_t maxVersion = 40, int32_t mask = -1, bool boostEcl = true);  // All optional parameters

		/*
		 * Creates a new QR Code with the given version number,
		 * error correction level, data codeword bytes, and mask number.
		 * This is a low-level API that most users should not use directly.
		 * A mid-level API is the encodeSegments() function.
		 */
		QrCode(int32_t ver, Ecc ecl, const std::vector<std::uint8_t>& dataCodewords, int32_t msk);

		/*---- Public instance methods ----*/

		/*
		 * Returns this QR Code's version, in the range [1, 40].
		 */
		int32_t getVersion() const;

		/*
		 * Returns this QR Code's size, in the range [21, 177].
		 */
		int32_t getSize() const;

		/*
		 * Returns this QR Code's error correction level.
		 */
		Ecc getErrorCorrectionLevel() const;

		/*
		 * Returns this QR Code's mask, in the range [0, 7].
		 */
		int32_t getMask() const;

		/*
		 * Returns the color of the module (pixel) at the given coordinates, which is false
		 * for light or true for dark. The top left corner has the coordinates (x=0, y=0).
		 * If the given coordinates are out of bounds, then false (light) is returned.
		 */
		bool getModule(int32_t x, int32_t y) const;

		// The minimum version number supported in the QR Code Model 2 standard.
		static constexpr int32_t MIN_VERSION = 1;

		// The maximum version number supported in the QR Code Model 2 standard.
		static constexpr int32_t MAX_VERSION = 40;

	private:
		// Returns a value in the range 0 to 3 (unsigned 2-bit integer).
		static int32_t getFormatBits(Ecc ecl);

		/* The version number of this QR Code, which is between 1 and 40 (inclusive).
		* This determines the size of this barcode. */
		int32_t m_version{ 0 };

		/* The width and height of this QR Code, measured in modules, between
		* 21 and 177 (inclusive). This is equal to version * 4 + 17. */
		int32_t m_size{};

		/* The error correction level used in this QR Code. */
		Ecc m_errorCorrectionLevel{};

		/* The index of the mask pattern used in this QR Code, which is between 0 and 7 (inclusive).
		* Even if a QR Code is created with automatic masking requested (mask = -1),
		* the resulting object still has a mask value between 0 and 7. */
		int32_t m_mask{ 0 };

		// Private grids of modules/pixels, with dimensions of size*size:

		// The modules of this QR Code (false = light, true = dark).
		// Immutable after constructor finishes. Accessed through getModule().
		std::vector<std::vector<bool> > m_modules{};

		// Indicates function modules that are not subjected to masking. Discarded when constructor finishes.
		std::vector<std::vector<bool>> m_isFunction{};

		// Reads this object's version field, and draws and marks all function modules.
		void drawFunctionPatterns();

		// Draws two copies of the format bits (with its own error correction code)
		// based on the given mask and this object's error correction level field.
		void drawFormatBits(int32_t msk);

		// Draws two copies of the version bits (with its own error correction code),
		// based on this object's version field, iff 7 <= version <= 40.
		void drawVersion();

		// Draws a 9*9 finder pattern including the border separator,
		// with the center module at (x, y). Modules can be out of bounds.
		void drawFinderPattern(int32_t x, int32_t y);

		// Draws a 5*5 alignment pattern, with the center module
		// at (x, y). All modules must be in bounds.
		void drawAlignmentPattern(int32_t x, int32_t y);

		// Sets the color of a module and marks it as a function module.
		// Only used by the constructor. Coordinates must be in bounds.
		void setFunctionModule(int32_t x, int32_t y, bool isDark);

		// Returns the color of the module at the given coordinates, which must be in range.
		bool get_module(int32_t x, int32_t y) const;

		/*---- Private helper methods for constructor: Codewords and masking ----*/

		// Returns a new byte string representing the given data with the appropriate error correction
		// codewords appended to it, based on this object's version and error correction level.
		std::vector<std::uint8_t> addEccAndInterleave(const std::vector<std::uint8_t>& data) const;

		// Draws the given sequence of 8-bit codewords (data and error correction) onto the entire
		// data area of this QR Code. Function modules need to be marked off before this is called.
		void drawCodewords(const std::vector<std::uint8_t>& data);

		// XORs the codeword modules in this QR Code with the given mask pattern.
		// The function modules must be marked and the codeword bits must be drawn
		// before masking. Due to the arithmetic of XOR, calling applyMask() with
		// the same mask value a second time will undo the mask. A final well-formed
		// QR Code needs exactly one (not zero, two, etc.) mask applied.
		void applyMask(int32_t msk);

		// Calculates and returns the penalty score based on state of this QR Code's current modules.
		// This is used by the automatic mask choice algorithm to find the mask pattern that yields the lowest score.
		long getPenaltyScore() const;

		/*---- Private helper functions ----*/

		// Returns an ascending list of positions of alignment patterns for this version number.
		// Each position is in the range [0,177), and are used on both the x and y axes.
		// This could be implemented as lookup table of 40 variable-length lists of unsigned bytes.
		std::vector<int32_t> getAlignmentPatternPositions() const;

		// Returns the number of data bits that can be stored in a QR Code of the given version number, after
		// all function modules are excluded. This includes remainder bits, so it might not be a multiple of 8.
		// The result is in the range [208, 29648]. This could be implemented as a 40-entry lookup table.
		static int32_t getNumRawDataModules(int32_t ver);

		// Returns the number of 8-bit data (i.e. not error correction) codewords contained in any
		// QR Code of the given version number and error correction level, with remainder bits discarded.
		// This stateless pure function could be implemented as a (40*4)-cell lookup table.
		static int32_t getNumDataCodewords(int32_t ver, Ecc ecl);

		// Returns a Reed-Solomon ECC generator polynomial for the given degree. This could be
		// implemented as a lookup table over all possible parameter values, instead of as an algorithm.
		static std::vector<std::uint8_t> reedSolomonComputeDivisor(int32_t degree);

		// Returns the Reed-Solomon error correction codeword for the given data and divisor polynomials.
		static std::vector<std::uint8_t> reedSolomonComputeRemainder(const std::vector<std::uint8_t>& data, const std::vector<std::uint8_t>& divisor);

		// Returns the product of the two given field elements modulo GF(2^8/0x11D).
		// All inputs are valid. This could be implemented as a 256*256 lookup table.
		static std::uint8_t reedSolomonMultiply(std::uint8_t x, std::uint8_t y);

		// Can only be called immediately after a light run is added, and
		// returns either 0, 1, or 2. A helper function for getPenaltyScore().
		int32_t finderPenaltyCountPatterns(const std::array<int32_t, 7>& runHistory) const;

		// Must be called at the end of a line (row or column) of modules. A helper function for getPenaltyScore().
		int32_t finderPenaltyTerminateAndCount(bool currentRunColor, int32_t currentRunLength, std::array<int32_t, 7>& runHistory) const;

		// Pushes the given value to the front and drops the last value. A helper function for getPenaltyScore().
		void finderPenaltyAddHistory(int32_t currentRunLength, std::array<int32_t, 7>& runHistory) const;

		// Returns true iff the i'th bit of x is set to 1.
		static bool getBit(long x, int32_t i);

		// For use in getPenaltyScore(), when evaluating which mask is best.
		static const int32_t PENALTY_N1;
		static const int32_t PENALTY_N2;
		static const int32_t PENALTY_N3;
		static const int32_t PENALTY_N4;

		static const std::array<std::array<int8_t, 41>, 4> ECC_CODEWORDS_PER_BLOCK;
		static const std::array<std::array<int8_t, 41>, 4> NUM_ERROR_CORRECTION_BLOCKS;
	};

	/*---- Public exception class ----*/

	/*
	 * Thrown when the supplied data does not fit any QR Code version. Ways to handle this exception include:
	 * - Decrease the error correction level if it was greater than Ecc::LOW.
	 * - If the encodeSegments() function was called with a maxVersion argument, then increase
	 *   it if it was less than QrCode::MAX_VERSION. (This advice does not apply to the other
	 *   factory functions because they search all versions up to QrCode::MAX_VERSION.)
	 * - Split the text data into better or optimal segments in order to reduce the number of bits required.
	 * - Change the text or binary data to be shorter.
	 * - Change the text to fit the character set of a particular segment mode (e.g. alphanumeric).
	 * - Propagate the error upward to the caller/user.
	 */
	class data_too_long : public std::length_error
	{
	public:
		explicit data_too_long(const std::string& msg);
	};

	/*
	 * An appendable sequence of bits (0s and 1s). Mainly used by QrSegment.
	 */
	class BitBuffer final : public std::vector<bool>
	{
	public:
		// Creates an empty bit buffer (length 0).
		BitBuffer();

		// Appends the given number of low-order bits of the given value
		// to this buffer. Requires 0 <= len <= 31 and val < 2^len.
		void appendBits(std::uint32_t val, int32_t len);
	};

	/* CODE SEGMENT *************************************************************/

	QrSegment::Mode::Mode(int32_t mode, int32_t cc0, int32_t cc1, int32_t cc2) : m_modeBits(mode)
	{
		m_numBitsCharCount[0] = cc0;
		m_numBitsCharCount[1] = cc1;
		m_numBitsCharCount[2] = cc2;
	}

	int32_t QrSegment::Mode::getModeBits() const
	{
		return m_modeBits;
	}

	int32_t QrSegment::Mode::numCharCountBits(int32_t ver) const
	{
		return m_numBitsCharCount[(ver + 7) / 17];
	}

	const QrSegment::Mode QrSegment::Mode::NUMERIC(0x1, 10, 12, 14);
	const QrSegment::Mode QrSegment::Mode::ALPHANUMERIC(0x2, 9, 11, 13);
	const QrSegment::Mode QrSegment::Mode::BYTE(0x4, 8, 16, 16);
	const QrSegment::Mode QrSegment::Mode::KANJI(0x8, 8, 10, 12);
	const QrSegment::Mode QrSegment::Mode::ECI(0x7, 0, 0, 0);

	QrSegment QrSegment::makeBytes(const std::vector<uint8_t>& data)
	{
		if (data.size() > static_cast<uint32_t>(std::numeric_limits<int32_t>::max()))
		{
			throw std::length_error("Data too long");
		}

		BitBuffer bb;

		for (uint8_t b : data)
		{
			bb.appendBits(b, 8);
		}

		return QrSegment(Mode::BYTE, static_cast<int32_t>(data.size()), std::move(bb));
	}

	QrSegment QrSegment::makeNumeric(const char* digits) {
		BitBuffer bb;
		int32_t accumData = 0;
		int32_t accumCount = 0;
		int32_t charCount = 0;
		for (; *digits != '\0'; digits++, charCount++) {
			char c = *digits;
			if (c < '0' || c > '9')
				throw std::domain_error("String contains non-numeric characters");
			accumData = accumData * 10 + (c - '0');
			accumCount++;
			if (accumCount == 3) {
				bb.appendBits(static_cast<uint32_t>(accumData), 10);
				accumData = 0;
				accumCount = 0;
			}
		}
		if (accumCount > 0)  // 1 or 2 digits remaining
			bb.appendBits(static_cast<uint32_t>(accumData), accumCount * 3 + 1);
		return QrSegment(Mode::NUMERIC, charCount, std::move(bb));
	}

	QrSegment QrSegment::makeAlphanumeric(const char* text) {
		BitBuffer bb;
		int32_t accumData = 0;
		int32_t accumCount = 0;
		int32_t charCount = 0;
		for (; *text != '\0'; text++, charCount++) {
			const char* temp = std::strchr(ALPHANUMERIC_CHARSET, *text);
			if (temp == nullptr)
				throw std::domain_error("String contains unencodable characters in alphanumeric mode");
			accumData = accumData * 45 + static_cast<int32_t>(temp - ALPHANUMERIC_CHARSET);
			accumCount++;
			if (accumCount == 2) {
				bb.appendBits(static_cast<uint32_t>(accumData), 11);
				accumData = 0;
				accumCount = 0;
			}
		}
		if (accumCount > 0)  // 1 character remaining
			bb.appendBits(static_cast<uint32_t>(accumData), 6);
		return QrSegment(Mode::ALPHANUMERIC, charCount, std::move(bb));
	}

	std::vector<QrSegment> QrSegment::makeSegments(const char* text) {
		// Select the most efficient segment encoding automatically
		std::vector<QrSegment> result;
		if (*text == '\0');  // Leave result empty
		else if (isNumeric(text))
			result.push_back(makeNumeric(text));
		else if (isAlphanumeric(text))
			result.push_back(makeAlphanumeric(text));
		else {
			std::vector<uint8_t> bytes;
			for (; *text != '\0'; text++)
				bytes.push_back(static_cast<uint8_t>(*text));
			result.push_back(makeBytes(bytes));
		}
		return result;
	}

	QrSegment QrSegment::makeEci(long assignVal) {
		BitBuffer bb;
		if (assignVal < 0)
			throw std::domain_error("ECI assignment value out of range");
		else if (assignVal < (1 << 7))
			bb.appendBits(static_cast<uint32_t>(assignVal), 8);
		else if (assignVal < (1 << 14)) {
			bb.appendBits(2, 2);
			bb.appendBits(static_cast<uint32_t>(assignVal), 14);
		}
		else if (assignVal < 1000000L) {
			bb.appendBits(6, 3);
			bb.appendBits(static_cast<uint32_t>(assignVal), 21);
		}
		else
			throw std::domain_error("ECI assignment value out of range");
		return QrSegment(Mode::ECI, 0, std::move(bb));
	}

	QrSegment::QrSegment(const Mode& md, int32_t numCh, const std::vector<bool>& dt) :
		m_mode(&md),
		m_numChars(numCh),
		m_data(dt) {
		if (numCh < 0)
			throw std::domain_error("Invalid value");
	}

	QrSegment::QrSegment(const Mode& md, int32_t numCh, std::vector<bool>&& dt) :
		m_mode(&md),
		m_numChars(numCh),
		m_data(std::move(dt)) {
		if (numCh < 0)
			throw std::domain_error("Invalid value");
	}

	int32_t QrSegment::getTotalBits(const std::vector<QrSegment>& segs, int32_t version) {
		int32_t result = 0;
		for (const QrSegment& seg : segs) {
			int32_t ccbits = seg.m_mode->numCharCountBits(version);
			if (seg.m_numChars >= (1L << ccbits))
				return -1;  // The segment's length doesn't fit the field's bit width
			if (4 + ccbits > std::numeric_limits<int32_t>::max() - result)
				return -1;  // The sum will overflow an int32_t type
			result += 4 + ccbits;
			if (seg.m_data.size() > static_cast<uint32_t>(std::numeric_limits<int32_t>::max() - result))
				return -1;  // The sum will overflow an int32_t type
			result += static_cast<int32_t>(seg.m_data.size());
		}
		return result;
	}

	bool QrSegment::isNumeric(const char* text) {
		for (; *text != '\0'; text++) {
			char c = *text;
			if (c < '0' || c > '9')
				return false;
		}
		return true;
	}

	bool QrSegment::isAlphanumeric(const char* text) {
		for (; *text != '\0'; text++) {
			if (std::strchr(ALPHANUMERIC_CHARSET, *text) == nullptr)
				return false;
		}
		return true;
	}

	const QrSegment::Mode& QrSegment::getMode() const {
		return *m_mode;
	}

	int32_t QrSegment::getNumChars() const {
		return m_numChars;
	}

	const std::vector<bool>& QrSegment::getData() const {
		return m_data;
	}

	const char* QrSegment::ALPHANUMERIC_CHARSET = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ $%*+-./:";

	/*---- Class QrCode ----*/

	int32_t QrCode::getFormatBits(Ecc ecl)
	{
		const std::array<int32_t, 4> output{ 1, 0, 3, 2 };
		return output[std::to_underlying(ecl)];
	}

	QrCode QrCode::encodeText(const char* text, Ecc ecl)
	{
		std::vector<QrSegment> segs = QrSegment::makeSegments(text);
		return encodeSegments(segs, ecl);
	}

	QrCode QrCode::encodeBinary(const std::vector<uint8_t>& data, Ecc ecl)
	{
		std::vector<QrSegment> segs{ QrSegment::makeBytes(data) };
		return encodeSegments(segs, ecl);
	}

	QrCode QrCode::encodeSegments(const std::vector<QrSegment>& segs, Ecc ecl, int32_t minVersion, int32_t maxVersion, int32_t mask, bool boostEcl)
	{
		if (!(MIN_VERSION <= minVersion && minVersion <= maxVersion && maxVersion <= MAX_VERSION) || mask < -1 || mask > 7)
			throw std::invalid_argument("Invalid value");

		// Find the minimal version number to use
		int32_t version{ 0 };
		int32_t dataUsedBits{ 0 };

		for (version = minVersion; ; version++)
		{
			int32_t dataCapacityBits = getNumDataCodewords(version, ecl) * 8;  // Number of data bits available
			dataUsedBits = QrSegment::getTotalBits(segs, version);
			if (dataUsedBits != -1 && dataUsedBits <= dataCapacityBits)
				break;  // This version number is found to be suitable
			if (version >= maxVersion) {  // All versions in the range could not fit the given data
				std::ostringstream sb;
				if (dataUsedBits == -1)
					sb << "Segment too long";
				else {
					sb << "Data length = " << dataUsedBits << " bits, ";
					sb << "Max capacity = " << dataCapacityBits << " bits";
				}
				throw data_too_long(sb.str());
			}
		}
		assert(dataUsedBits != -1);

		// Increase the error correction level while the data still fits in the current version number
		for (Ecc newEcl : {Ecc::MEDIUM, Ecc::QUARTILE, Ecc::HIGH}) {  // From low to high
			if (boostEcl && dataUsedBits <= getNumDataCodewords(version, newEcl) * 8)
				ecl = newEcl;
		}

		// Concatenate all segments to create the data bit string
		BitBuffer bb;
		for (const QrSegment& seg : segs) {
			bb.appendBits(static_cast<uint32_t>(seg.getMode().getModeBits()), 4);
			bb.appendBits(static_cast<uint32_t>(seg.getNumChars()), seg.getMode().numCharCountBits(version));
			bb.insert(bb.end(), seg.getData().begin(), seg.getData().end());
		}
		assert(bb.size() == static_cast<uint32_t>(dataUsedBits));

		// Add terminator and pad up to a byte if applicable
		size_t dataCapacityBits = static_cast<size_t>(getNumDataCodewords(version, ecl)) * 8;
		assert(bb.size() <= dataCapacityBits);
		bb.appendBits(0, std::min(4, static_cast<int32_t>(dataCapacityBits - bb.size())));
		bb.appendBits(0, (8 - static_cast<int32_t>(bb.size() % 8)) % 8);
		assert(bb.size() % 8 == 0);

		// Pad with alternating bytes until data capacity is reached
		for (uint8_t padByte = 0xEC; bb.size() < dataCapacityBits; padByte ^= 0xEC ^ 0x11)
			bb.appendBits(padByte, 8);

		// Pack bits into bytes in big endian
		std::vector<uint8_t> dataCodewords(bb.size() / 8);
		for (size_t i = 0; i < bb.size(); i++)
			dataCodewords.at(i >> 3) |= (bb.at(i) ? 1 : 0) << (7 - (i & 7));

		// Create the QR Code object
		return QrCode(version, ecl, dataCodewords, mask);
	}

	QrCode::QrCode(int32_t ver, Ecc ecl, const std::vector<uint8_t>& dataCodewords, int32_t msk) :
		// Initialize fields and check arguments
		m_version(ver),
		m_errorCorrectionLevel(ecl) {
		if (ver < MIN_VERSION || ver > MAX_VERSION)
			throw std::domain_error("Version value out of range");
		if (msk < -1 || msk > 7)
			throw std::domain_error("Mask value out of range");
		m_size = ver * 4 + 17;
		auto sz = static_cast<size_t>(m_size);
		m_modules = std::vector<std::vector<bool> >(sz, std::vector<bool>(sz));  // Initially all light
		m_isFunction = std::vector<std::vector<bool> >(sz, std::vector<bool>(sz));

		// Compute ECC, draw modules
		drawFunctionPatterns();
		const std::vector<uint8_t> allCodewords = addEccAndInterleave(dataCodewords);
		drawCodewords(allCodewords);

		// Do masking
		if (msk == -1) {  // Automatically choose best mask
			long minPenalty = LONG_MAX;
			for (int32_t i = 0; i < 8; i++) {
				applyMask(i);
				drawFormatBits(i);
				long penalty = getPenaltyScore();
				if (penalty < minPenalty) {
					msk = i;
					minPenalty = penalty;
				}
				applyMask(i);  // Undoes the mask due to XOR
			}
		}
		assert(0 <= msk && msk <= 7);
		m_mask = msk;
		applyMask(msk);  // Apply the final choice of mask
		drawFormatBits(msk);  // Overwrite old format bits

		m_isFunction.clear();
		m_isFunction.shrink_to_fit();
	}

	int32_t QrCode::getVersion() const {
		return m_version;
	}

	int32_t QrCode::getSize() const {
		return m_size;
	}

	QrCode::Ecc QrCode::getErrorCorrectionLevel() const {
		return m_errorCorrectionLevel;
	}

	int32_t QrCode::getMask() const {
		return m_mask;
	}

	bool QrCode::getModule(int32_t x, int32_t y) const {
		return 0 <= x && x < m_size && 0 <= y && y < m_size && get_module(x, y);
	}

	void QrCode::drawFunctionPatterns() {
		// Draw horizontal and vertical timing patterns
		for (int32_t i = 0; i < m_size; i++) {
			setFunctionModule(6, i, i % 2 == 0);
			setFunctionModule(i, 6, i % 2 == 0);
		}

		// Draw 3 finder patterns (all corners except bottom right; overwrites some timing modules)
		drawFinderPattern(3, 3);
		drawFinderPattern(m_size - 4, 3);
		drawFinderPattern(3, m_size - 4);

		// Draw numerous alignment patterns
		const std::vector<int32_t> alignPatPos = getAlignmentPatternPositions();
		size_t numAlign = alignPatPos.size();
		for (size_t i = 0; i < numAlign; i++) {
			for (size_t j = 0; j < numAlign; j++) {
				// Don't draw on the three finder corners
				if (!((i == 0 && j == 0) || (i == 0 && j == numAlign - 1) || (i == numAlign - 1 && j == 0)))
					drawAlignmentPattern(alignPatPos.at(i), alignPatPos.at(j));
			}
		}

		// Draw configuration data
		drawFormatBits(0);  // Dummy mask value; overwritten later in the constructor
		drawVersion();
	}

	void QrCode::drawFormatBits(int32_t msk) {
		// Calculate error correction code and pack bits
		int32_t data = getFormatBits(m_errorCorrectionLevel) << 3 | msk;  // errCorrLvl is uint2, mask is uint3
		int32_t rem = data;
		for (int32_t i = 0; i < 10; i++)
			rem = (rem << 1) ^ ((rem >> 9) * 0x537);
		int32_t bits = (data << 10 | rem) ^ 0x5412;  // uint15
		assert(bits >> 15 == 0);

		// Draw first copy
		for (int32_t i = 0; i <= 5; i++)
			setFunctionModule(8, i, getBit(bits, i));
		setFunctionModule(8, 7, getBit(bits, 6));
		setFunctionModule(8, 8, getBit(bits, 7));
		setFunctionModule(7, 8, getBit(bits, 8));
		for (int32_t i = 9; i < 15; i++)
			setFunctionModule(14 - i, 8, getBit(bits, i));

		// Draw second copy
		for (int32_t i = 0; i < 8; i++)
			setFunctionModule(m_size - 1 - i, 8, getBit(bits, i));
		for (int32_t i = 8; i < 15; i++)
			setFunctionModule(8, m_size - 15 + i, getBit(bits, i));
		setFunctionModule(8, m_size - 8, true);  // Always dark
	}

	void QrCode::drawVersion() {
		if (m_version < 7)
			return;

		// Calculate error correction code and pack bits
		int32_t rem = m_version;  // version is uint6, in the range [7, 40]
		for (int32_t i = 0; i < 12; i++)
			rem = (rem << 1) ^ ((rem >> 11) * 0x1F25);
		long bits = static_cast<long>(m_version) << 12 | rem;  // uint18
		assert(bits >> 18 == 0);

		// Draw two copies
		for (int32_t i = 0; i < 18; i++) {
			bool bit = getBit(bits, i);
			int32_t a = m_size - 11 + i % 3;
			int32_t b = i / 3;
			setFunctionModule(a, b, bit);
			setFunctionModule(b, a, bit);
		}
	}

	void QrCode::drawFinderPattern(int32_t x, int32_t y) {
		for (int32_t dy = -4; dy <= 4; dy++) {
			for (int32_t dx = -4; dx <= 4; dx++) {
				int32_t dist = std::max(std::abs(dx), std::abs(dy));  // Chebyshev/infinity norm
				int32_t xx = x + dx;
				int32_t yy = y + dy;
				if (0 <= xx && xx < m_size && 0 <= yy && yy < m_size)
					setFunctionModule(xx, yy, dist != 2 && dist != 4);
			}
		}
	}

	void QrCode::drawAlignmentPattern(int32_t x, int32_t y) {
		for (int32_t dy = -2; dy <= 2; dy++) {
			for (int32_t dx = -2; dx <= 2; dx++)
				setFunctionModule(x + dx, y + dy, std::max(std::abs(dx), std::abs(dy)) != 1);
		}
	}

	void QrCode::setFunctionModule(int32_t x, int32_t y, bool isDark) {
		auto ux = static_cast<size_t>(x);
		auto uy = static_cast<size_t>(y);
		m_modules.at(uy).at(ux) = isDark;
		m_isFunction.at(uy).at(ux) = true;
	}

	bool QrCode::get_module(int32_t x, int32_t y) const {
		return m_modules.at(static_cast<size_t>(y)).at(static_cast<size_t>(x));
	}

	std::vector<uint8_t> QrCode::addEccAndInterleave(const std::vector<uint8_t>& data) const {
		if (data.size() != static_cast<uint32_t>(getNumDataCodewords(m_version, m_errorCorrectionLevel)))
			throw std::invalid_argument("Invalid argument");

		// Calculate parameter numbers
		int32_t numBlocks = NUM_ERROR_CORRECTION_BLOCKS[static_cast<int32_t>(m_errorCorrectionLevel)][m_version];
		int32_t blockEccLen = ECC_CODEWORDS_PER_BLOCK[static_cast<int32_t>(m_errorCorrectionLevel)][m_version];
		int32_t rawCodewords = getNumRawDataModules(m_version) / 8;
		int32_t numShortBlocks = numBlocks - rawCodewords % numBlocks;
		int32_t shortBlockLen = rawCodewords / numBlocks;

		// Split data into blocks and append ECC to each block
		std::vector<std::vector<uint8_t> > blocks;
		const std::vector<uint8_t> rsDiv = reedSolomonComputeDivisor(blockEccLen);
		for (int32_t i = 0, k = 0; i < numBlocks; i++) {
			std::vector<uint8_t> dat(data.cbegin() + k, data.cbegin() + (k + shortBlockLen - blockEccLen + (i < numShortBlocks ? 0 : 1)));
			k += static_cast<int32_t>(dat.size());
			const std::vector<uint8_t> ecc = reedSolomonComputeRemainder(dat, rsDiv);
			if (i < numShortBlocks)
				dat.push_back(0);
			dat.insert(dat.end(), ecc.cbegin(), ecc.cend());
			blocks.push_back(std::move(dat));
		}

		// Interleave (not concatenate) the bytes from every block into a single sequence
		std::vector<uint8_t> result;
		for (size_t i = 0; i < blocks.at(0).size(); i++) {
			for (size_t j = 0; j < blocks.size(); j++) {
				// Skip the padding byte in short blocks
				if (i != static_cast<uint32_t>(shortBlockLen - blockEccLen) || j >= static_cast<uint32_t>(numShortBlocks))
					result.push_back(blocks.at(j).at(i));
			}
		}
		assert(result.size() == static_cast<uint32_t>(rawCodewords));
		return result;
	}

	void QrCode::drawCodewords(const std::vector<uint8_t>& data) {
		if (data.size() != static_cast<uint32_t>(getNumRawDataModules(m_version) / 8))
			throw std::invalid_argument("Invalid argument");

		size_t i = 0;  // Bit index into the data
		// Do the funny zigzag scan
		for (int32_t right = m_size - 1; right >= 1; right -= 2)
		{
			// Index of right column in each column pair
			if (right == 6)
				right = 5;

			for (int32_t vert = 0; vert < m_size; vert++)
			{
				// Vertical counter
				for (int32_t j = 0; j < 2; j++)
				{
					auto x = static_cast<size_t>(right - j);  // Actual x coordinate
					bool upward = ((right + 1) & 2) == 0;
					auto y = static_cast<size_t>(upward ? m_size - 1 - vert : vert);  // Actual y coordinate

					if (!m_isFunction.at(y).at(x) && i < data.size() * 8)
					{
						m_modules.at(y).at(x) = getBit(data.at(i >> 3), 7 - static_cast<int32_t>(i & 7));
						i++;
					}
					// If this QR Code has any remainder bits (0 to 7), they were assigned as
					// 0/false/light by the constructor and are left unchanged by this method
				}
			}
		}
		assert(i == data.size() * 8);
	}

	void QrCode::applyMask(int32_t msk) {
		if (msk < 0 || msk > 7)
			throw std::domain_error("Mask value out of range");
		auto sz = static_cast<size_t>(m_size);
		for (size_t y = 0; y < sz; y++) {
			for (size_t x = 0; x < sz; x++) {
				bool invert;
				switch (msk) {
				case 0:  invert = (x + y) % 2 == 0;                    break;
				case 1:  invert = y % 2 == 0;                          break;
				case 2:  invert = x % 3 == 0;                          break;
				case 3:  invert = (x + y) % 3 == 0;                    break;
				case 4:  invert = (x / 3 + y / 2) % 2 == 0;            break;
				case 5:  invert = x * y % 2 + x * y % 3 == 0;          break;
				case 6:  invert = (x * y % 2 + x * y % 3) % 2 == 0;    break;
				case 7:  invert = ((x + y) % 2 + x * y % 3) % 2 == 0;  break;
				default:  throw std::logic_error("Unreachable");
				}
				m_modules.at(y).at(x) = m_modules.at(y).at(x) ^ (invert & !m_isFunction.at(y).at(x));
			}
		}
	}

	long QrCode::getPenaltyScore() const
	{
		long result = 0;

		// Adjacent modules in row having same color, and finder-like patterns
		for (int32_t y = 0; y < m_size; y++) {
			bool runColor = false;
			int32_t runX = 0;
			std::array<int32_t, 7> runHistory = {};
			for (int32_t x = 0; x < m_size; x++) {
				if (get_module(x, y) == runColor) {
					runX++;
					if (runX == 5)
						result += PENALTY_N1;
					else if (runX > 5)
						result++;
				}
				else {
					finderPenaltyAddHistory(runX, runHistory);
					if (!runColor)
						result += finderPenaltyCountPatterns(runHistory) * PENALTY_N3;
					runColor = get_module(x, y);
					runX = 1;
				}
			}
			result += finderPenaltyTerminateAndCount(runColor, runX, runHistory) * PENALTY_N3;
		}
		// Adjacent modules in column having same color, and finder-like patterns
		for (int32_t x = 0; x < m_size; x++) {
			bool runColor = false;
			int32_t runY = 0;
			std::array<int32_t, 7> runHistory = {};
			for (int32_t y = 0; y < m_size; y++) {
				if (get_module(x, y) == runColor) {
					runY++;
					if (runY == 5)
						result += PENALTY_N1;
					else if (runY > 5)
						result++;
				}
				else {
					finderPenaltyAddHistory(runY, runHistory);
					if (!runColor)
						result += finderPenaltyCountPatterns(runHistory) * PENALTY_N3;
					runColor = get_module(x, y);
					runY = 1;
				}
			}
			result += finderPenaltyTerminateAndCount(runColor, runY, runHistory) * PENALTY_N3;
		}

		// 2*2 blocks of modules having same color
		for (int32_t y = 0; y < m_size - 1; y++) {
			for (int32_t x = 0; x < m_size - 1; x++) {
				bool  color = get_module(x, y);
				if (color == get_module(x + 1, y) &&
					color == get_module(x, y + 1) &&
					color == get_module(x + 1, y + 1))
					result += PENALTY_N2;
			}
		}

		// Balance of dark and light modules
		int32_t dark = 0;
		for (const std::vector<bool>& row : m_modules) {
			for (bool color : row) {
				if (color)
					dark++;
			}
		}
		int32_t total = m_size * m_size;  // Note that size is odd, so dark/total != 1/2
		// Compute the smallest integer k >= 0 such that (45-5k)% <= dark/total <= (55+5k)%
		int32_t k = static_cast<int32_t>((std::abs(dark * 20L - total * 10L) + total - 1) / total) - 1;
		assert(0 <= k && k <= 9);
		result += k * PENALTY_N4;
		assert(0 <= result && result <= 2568888L);  // Non-tight upper bound based on default values of PENALTY_N1, ..., N4
		return result;
	}

	std::vector<int32_t> QrCode::getAlignmentPatternPositions() const
	{
		if (m_version == 1)
			return std::vector<int32_t>();
		else {
			int32_t numAlign = m_version / 7 + 2;
			int32_t step = (m_version * 8 + numAlign * 3 + 5) / (numAlign * 4 - 4) * 2;
			std::vector<int32_t> result;
			for (int32_t i = 0, pos = m_size - 7; i < numAlign - 1; i++, pos -= step)
				result.insert(result.begin(), pos);
			result.insert(result.begin(), 6);
			return result;
		}
	}

	int32_t QrCode::getNumRawDataModules(int32_t ver) {
		if (ver < MIN_VERSION || ver > MAX_VERSION)
			throw std::domain_error("Version number out of range");
		int32_t result = (16 * ver + 128) * ver + 64;
		if (ver >= 2) {
			int32_t numAlign = ver / 7 + 2;
			result -= (25 * numAlign - 10) * numAlign - 55;
			if (ver >= 7)
				result -= 36;
		}
		assert(208 <= result && result <= 29648);
		return result;
	}

	int32_t QrCode::getNumDataCodewords(int32_t ver, Ecc ecl) {
		return getNumRawDataModules(ver) / 8
			- ECC_CODEWORDS_PER_BLOCK[static_cast<int32_t>(ecl)][ver]
			* NUM_ERROR_CORRECTION_BLOCKS[static_cast<int32_t>(ecl)][ver];
	}

	std::vector<uint8_t> QrCode::reedSolomonComputeDivisor(int32_t degree) {
		if (degree < 1 || degree > 255)
			throw std::domain_error("Degree out of range");
		// Polynomial coefficients are stored from highest to lowest power, excluding the leading term which is always 1.
		// For example the polynomial x^3 + 255x^2 + 8x + 93 is stored as the uint8 array {255, 8, 93}.
		std::vector<uint8_t> result(static_cast<size_t>(degree));
		result.at(result.size() - 1) = 1;  // Start off with the monomial x^0

		// Compute the product polynomial (x - r^0) * (x - r^1) * (x - r^2) * ... * (x - r^{degree-1}),
		// and drop the highest monomial term which is always 1x^degree.
		// Note that r = 0x02, which is a generator element of this field GF(2^8/0x11D).
		uint8_t root = 1;
		for (int32_t i = 0; i < degree; i++) {
			// Multiply the current product by (x - r^i)
			for (size_t j = 0; j < result.size(); j++) {
				result.at(j) = reedSolomonMultiply(result.at(j), root);
				if (j + 1 < result.size())
					result.at(j) ^= result.at(j + 1);
			}
			root = reedSolomonMultiply(root, 0x02);
		}
		return result;
	}

	std::vector<uint8_t> QrCode::reedSolomonComputeRemainder(const std::vector<uint8_t>& data, const std::vector<uint8_t>& divisor) {
		std::vector<uint8_t> result(divisor.size());
		for (uint8_t b : data) {  // Polynomial division
			uint8_t factor = b ^ result.at(0);
			result.erase(result.begin());
			result.push_back(0);
			for (size_t i = 0; i < result.size(); i++)
				result.at(i) ^= reedSolomonMultiply(divisor.at(i), factor);
		}
		return result;
	}

	uint8_t QrCode::reedSolomonMultiply(uint8_t x, uint8_t y) {
		// Russian peasant multiplication
		int32_t z = 0;
		for (int32_t i = 7; i >= 0; i--) {
			z = (z << 1) ^ ((z >> 7) * 0x11D);
			z ^= ((y >> i) & 1) * x;
		}
		assert(z >> 8 == 0);
		return static_cast<uint8_t>(z);
	}

	int32_t QrCode::finderPenaltyCountPatterns(const std::array<int32_t, 7>& runHistory) const {
		int32_t n = runHistory.at(1);
		assert(n <= m_size * 3);
		bool core = n > 0 && runHistory.at(2) == n && runHistory.at(3) == n * 3 && runHistory.at(4) == n && runHistory.at(5) == n;
		return (core && runHistory.at(0) >= n * 4 && runHistory.at(6) >= n ? 1 : 0)
			+ (core && runHistory.at(6) >= n * 4 && runHistory.at(0) >= n ? 1 : 0);
	}

	int32_t QrCode::finderPenaltyTerminateAndCount(bool currentRunColor, int32_t currentRunLength, std::array<int32_t, 7>& runHistory) const {
		if (currentRunColor) {  // Terminate dark run
			finderPenaltyAddHistory(currentRunLength, runHistory);
			currentRunLength = 0;
		}
		currentRunLength += m_size;  // Add light border to final run
		finderPenaltyAddHistory(currentRunLength, runHistory);
		return finderPenaltyCountPatterns(runHistory);
	}

	void QrCode::finderPenaltyAddHistory(int32_t currentRunLength, std::array<int32_t, 7>& runHistory) const {
		if (runHistory.at(0) == 0)
			currentRunLength += m_size;  // Add light border to initial run
		std::copy_backward(runHistory.cbegin(), runHistory.cend() - 1, runHistory.end());
		runHistory.at(0) = currentRunLength;
	}

	bool QrCode::getBit(long x, int32_t i) {
		return ((x >> i) & 1) != 0;
	}

	/*---- Tables of constants ----*/

	const int32_t QrCode::PENALTY_N1 = 3;
	const int32_t QrCode::PENALTY_N2 = 3;
	const int32_t QrCode::PENALTY_N3 = 40;
	const int32_t QrCode::PENALTY_N4 = 10;

	const std::array<std::array<int8_t, 41>, 4> QrCode::ECC_CODEWORDS_PER_BLOCK
	{
		{
			// Version: (note that index 0 is for padding, and is set to an illegal value)
			//0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40    Error correction level
			{-1,  7, 10, 15, 20, 26, 18, 20, 24, 30, 18, 20, 24, 26, 30, 22, 24, 28, 30, 28, 28, 28, 28, 30, 30, 26, 28, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30},  // Low
			{-1, 10, 16, 26, 18, 24, 16, 18, 22, 22, 26, 30, 22, 22, 24, 24, 28, 28, 26, 26, 26, 26, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28},  // Medium
			{-1, 13, 22, 18, 26, 18, 24, 18, 22, 20, 24, 28, 26, 24, 20, 30, 24, 28, 28, 26, 30, 28, 30, 30, 30, 30, 28, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30},  // Quartile
			{-1, 17, 28, 22, 16, 22, 28, 26, 26, 24, 28, 24, 28, 22, 24, 24, 30, 28, 28, 26, 28, 30, 24, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30},  // High
		}
	};

	const std::array<std::array<int8_t, 41>, 4> QrCode::NUM_ERROR_CORRECTION_BLOCKS
	{
		{
			// Version: (note that index 0 is for padding, and is set to an illegal value)
			//0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40    Error correction level
			{-1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 4, 4, 4, 4, 4, 6, 6, 6, 6, 7, 8, 8, 9, 9, 10, 12, 12, 12, 13, 14, 15, 16, 17, 18, 19, 19, 20, 21, 22, 24, 25},  // Low
			{ -1, 1, 1, 1, 2, 2, 4, 4, 4, 5, 5,  5,  8,  9,  9, 10, 10, 11, 13, 14, 16, 17, 17, 18, 20, 21, 23, 25, 26, 28, 29, 31, 33, 35, 37, 38, 40, 43, 45, 47, 49 },  // Medium
			{ -1, 1, 1, 2, 2, 4, 4, 6, 6, 8, 8,  8, 10, 12, 16, 12, 17, 16, 18, 21, 20, 23, 23, 25, 27, 29, 34, 34, 35, 38, 40, 43, 45, 48, 51, 53, 56, 59, 62, 65, 68 },  // Quartile
			{ -1, 1, 1, 2, 4, 4, 4, 5, 6, 8, 8, 11, 11, 16, 16, 18, 16, 19, 21, 25, 25, 25, 34, 30, 32, 35, 37, 40, 42, 45, 48, 51, 54, 57, 60, 63, 66, 70, 74, 77, 81 },  // High
		}
	};

	data_too_long::data_too_long(const std::string& msg) :
		std::length_error(msg) {}

	/*---- Class BitBuffer ----*/

	BitBuffer::BitBuffer()
		: std::vector<bool>() {}

	void BitBuffer::appendBits(std::uint32_t val, int32_t len) {
		if (len < 0 || len > 31 || val >> len != 0)
			throw std::domain_error("Value out of range");
		for (int32_t i = len - 1; i >= 0; i--)  // Append bit by bit
			this->push_back(((val >> i) & 1) != 0);
	}

	ALLEGRO::BITMAP generate(const QrCode& qr, size_t border)
	{
		ALLEGRO::BITMAP target{ al::get_target_bitmap() };
		ALLEGRO::BITMAP out{};
		int32_t size = qr.getSize();

		out = al::create_bitmap({ size + border * 2, size + border * 2 });

		if (!out)
		{
			return nullptr;
		}

		al::set_target_bitmap(out);
		al::clear_to_color(wind::map_rgba_i(0xffffffff));

		ALLEGRO::BITMAP_LOCKED_REGION region = al::lock_bitmap(out, ALLEGRO::BITMAP_LOCK_WRITE_ONLY, ALLEGRO::PIXEL_FORMAT_ANY);

		for (int32_t y = 0; y < size; ++y)
		{
			for (int32_t x = 0; x < size; ++x)
			{
				if (qr.getModule(x, y))
				{
					al::put_pixel({ border + x, border + y }, wind::map_rgb_i(0x0));
				}
			}
		}

		al::unlock_bitmap(out);
		al::set_target_bitmap(target);

		return out;
	}

	ALLEGRO::BITMAP generate(const wind::string_t& text, size_t border)
	{
		if (text.size())
		{
			const QrCode qr0 = QrCode::encodeText(text.c_str(), QrCode::Ecc::MEDIUM);
			return generate(qr0, border);
		}

		return ALLEGRO::BITMAP();
	}
}