module game;

import <cstdint>;
import <array>;
import allegro;
import allegro.primitives_addon;
import wind;
import :base;
import :dungeon.base;
import :dungeon.map.outline;

namespace game
{
	namespace dungeon
	{
		namespace map
		{
			namespace outline
			{
				auto create_room(bool standard = false) -> room_t
				{
					ALLEGRO::SIZE<int32_t> room_size
					{
						((int32_t)wind::random::generate(GAME::DUNGEON::OUTLINE::MINIMUM.width, GAME::DUNGEON::OUTLINE::MAXIMUM.width) << 1) + 1,
						((int32_t)wind::random::generate(GAME::DUNGEON::OUTLINE::MINIMUM.height, GAME::DUNGEON::OUTLINE::MAXIMUM.height) << 1) + 1
					};

					if (standard)
					{
						room_size = GAME::DUNGEON::CAMERA::VIEWPORT;
					}

					return room_t{ 0, 0, room_size.width - 1, room_size.height - 1 };
				}

				auto place_room(room_t& room, GAME::DIRECTION direction, const ALLEGRO::POINT<int32_t>& touching_point)
				{
					ALLEGRO::POINT<int32_t> delta{ 0, 0 };

					switch (direction)
					{
					case GAME::DIRECTION::NORTH:
					{
						delta.x = (touching_point.x - (room.get_width() >> 1)) - room.top_left.x;
						delta.y = touching_point.y - room.bottom_right.y;
					} break;

					case GAME::DIRECTION::EAST:
					{
						delta.x = touching_point.x - room.top_left.x;
						delta.y = (touching_point.y - (room.get_height() >> 1)) - room.top_left.y;
					} break;

					case GAME::DIRECTION::SOUTH:
					{
						delta.x = (touching_point.x - (room.get_width() >> 1)) - room.top_left.x;
						delta.y = touching_point.y - room.top_left.y;
					} break;

					case GAME::DIRECTION::WEST:
					{
						delta.x = touching_point.x - room.bottom_right.x;
						delta.y = (touching_point.y - (room.get_height() >> 1)) - room.top_left.y;
					} break;
					}

					room.top_left.x += delta.x;
					room.top_left.y += delta.y;
					room.bottom_right.x += delta.x;
					room.bottom_right.y += delta.y;
				}

				auto collide(const room_t& room1, const room_t& room2) -> bool
				{
					int32_t room1_w = room1.get_width();
					int32_t room1_h = room1.get_height();
					int32_t room2_w = room2.get_width();
					int32_t room2_h = room2.get_height();

					return (room1.top_left.x < room2.top_left.x + room2_w &&
						room1.top_left.x + room1_w > room2.top_left.x &&
						room1.top_left.y < room2.top_left.y + room2_h &&
						room1.top_left.y + room1_h > room2.top_left.y);
				}

				auto has_collision(const outline_t& outline, const room_t& last_room, const room_t& new_room) -> bool
				{
					room_t over{};

					for (auto it = outline.m_rooms.cbegin(); it != outline.m_rooms.cend(); ++it)
					{
						if (&last_room == &(*it))
						{
							continue;
						}

						if (collide(*it, new_room))
						{
							return true;
						}
					}

					return false;
				}

				auto randomize_direction_array(wind::array_t<GAME::DIRECTION>& array) -> void
				{
					size_t a{ 0 };
					size_t b{ 0 };

					for (size_t i = 0; i < GAME::DUNGEON::OUTLINE::SHUFFLE_COUNT; ++i)
					{
						a = wind::random::generate((int32_t)GAME::DIRECTION::COUNT);
						while (a == b)
						{
							b = wind::random::generate((int32_t)GAME::DIRECTION::COUNT);
						}

						wind::array::swap(array, a, b);
					}
				}

				auto attach_room(outline_t& outline, const room_t& last_room, size_t count, bool standard = false) -> bool
				{
					ALLEGRO::POINT<int32_t> door{ 0, 0 };
					wind::array_t<GAME::DIRECTION> directions{ GAME::DIRECTION::NORTH, GAME::DIRECTION::EAST, GAME::DIRECTION::SOUTH, GAME::DIRECTION::WEST };
					room_t new_room = create_room(standard ? (outline.m_rooms.size() < (count - 1) ? false : true) : false);
					ALLEGRO::SIZE<int32_t> size{ new_room.get_width(), new_room.get_height() };

					randomize_direction_array(directions);

					for (size_t i = 0; i < (size_t)GAME::DIRECTION::COUNT; ++i)
					{
						GAME::DIRECTION direction = directions[i];

						switch (direction)
						{
						case GAME::DIRECTION::NORTH:
						{
							door = { (int32_t)wind::random::generate(last_room.top_left.x + 1, last_room.bottom_right.x - 1), last_room.top_left.y };
							new_room.top_left.x = door.x - (size.width >> 1);
							new_room.top_left.y = door.y - size.height;
							new_room.top_left.x = std::min(std::max(last_room.bottom_right.x - 1, new_room.top_left.x + wind::random::generate(1 + (GAME::DUNGEON::OUTLINE::DOOR_VARIANCE << 1)) - GAME::DUNGEON::OUTLINE::DOOR_VARIANCE), last_room.top_left.x + 1);
						} break;

						case GAME::DIRECTION::EAST:
						{
							door = { last_room.bottom_right.x, (int32_t)wind::random::generate(last_room.top_left.y + 1, last_room.bottom_right.y - 1) };
							new_room.top_left.x = door.x;
							new_room.top_left.y = door.y - (size.height >> 1);
							new_room.top_left.y = std::min(std::max(last_room.bottom_right.y - 1, new_room.top_left.y + wind::random::generate(1 + (GAME::DUNGEON::OUTLINE::DOOR_VARIANCE << 1)) - GAME::DUNGEON::OUTLINE::DOOR_VARIANCE), last_room.top_left.y + 1);
						} break;

						case GAME::DIRECTION::SOUTH:
						{
							door = { (int32_t)wind::random::generate(last_room.top_left.x + 1, last_room.bottom_right.x - 1), last_room.bottom_right.y };

							new_room.top_left.x = door.x - (size.width >> 1);
							new_room.top_left.y = door.y;
							new_room.top_left.x = std::min(std::max(last_room.bottom_right.x - 1, new_room.top_left.x + wind::random::generate(1 + (GAME::DUNGEON::OUTLINE::DOOR_VARIANCE << 1)) - GAME::DUNGEON::OUTLINE::DOOR_VARIANCE), last_room.top_left.x + 1);
						} break;

						case GAME::DIRECTION::WEST:
						{
							door = { last_room.top_left.x, (int32_t)wind::random::generate(last_room.top_left.y + 1, last_room.bottom_right.y - 1) };
							new_room.top_left.x = door.x - size.width;
							new_room.top_left.y = door.y - (size.height >> 1);
							new_room.top_left.y = std::min(std::max(last_room.bottom_right.y - 1, new_room.top_left.y + wind::random::generate(1 + (GAME::DUNGEON::OUTLINE::DOOR_VARIANCE << 1)) - GAME::DUNGEON::OUTLINE::DOOR_VARIANCE), last_room.top_left.y + 1);
						} break;
						}

						new_room.bottom_right.x = new_room.top_left.x + size.width - 1;
						new_room.bottom_right.y = new_room.top_left.y + size.height - 1;

						place_room(new_room, direction, door);
						if (!has_collision(outline, last_room, new_room))
						{
							outline.m_rooms.push_back(new_room);
							outline.m_doors.push_back(door);
							return true;
						}
					}

					return false;
				}

				auto calculate_min_max(const outline_t& outline, ALLEGRO::POINT<int32_t>& min_point, ALLEGRO::POINT<int32_t>& max_point) -> void
				{
					for (auto it = outline.m_rooms.cbegin(); it != outline.m_rooms.cend(); ++it)
					{
						min_point.x = std::min(min_point.x, (*it).top_left.x);
						min_point.y = std::min(min_point.y, (*it).top_left.y);
						max_point.x = std::max(max_point.x, (*it).bottom_right.x);
						max_point.y = std::max(max_point.y, (*it).bottom_right.y);
					}
				}

				auto clean_up(outline_t& outline)
				{
					ALLEGRO::POINT<int32_t> min_point{ 1000, 1000 };
					ALLEGRO::POINT<int32_t> max_point{ -1000, -1000 };

					calculate_min_max(outline, min_point, max_point);

					outline.m_size.width = (max_point.x - min_point.x + 1) + (GAME::DUNGEON::OUTLINE::MAP_PADDING << 1);
					outline.m_size.height = (max_point.y - min_point.y + 1) + (GAME::DUNGEON::OUTLINE::MAP_PADDING << 1);

					min_point.x -= GAME::DUNGEON::OUTLINE::MAP_PADDING;
					min_point.y -= GAME::DUNGEON::OUTLINE::MAP_PADDING;

					for (auto it = outline.m_rooms.begin(); it != outline.m_rooms.end(); ++it)
					{
						(*it).top_left.x -= min_point.x;
						(*it).top_left.y -= min_point.y;
						(*it).bottom_right.x -= min_point.x;
						(*it).bottom_right.y -= min_point.y;
					}

					for (auto it = outline.m_doors.begin(); it != outline.m_doors.end(); ++it)
					{
						(*it).x -= min_point.x;
						(*it).y -= min_point.y;
					}
				}

				auto generate(outline_t& outline, size_t direct_count, size_t extra_count) -> int32_t
				{
					size_t count{ 0 };

					outline.m_rooms.clear();
					outline.m_doors.clear();

					if (direct_count < 2)
					{
						return -1;
					}

					room_t room = create_room(true);

					outline.m_rooms.push_back(room);

					count = direct_count;

					while (outline.m_rooms.size() < count)
					{
						int32_t last_room{ (int32_t)outline.m_rooms.size() - 1 };

						while (last_room >= 0 && !attach_room(outline, outline.m_rooms[last_room], count - 1, true))
						{
							--last_room;
						}

						if (last_room < 0)
						{
							last_room = (int32_t)outline.m_rooms.size() - 1;
						}
					}

					int32_t current_size = (int32_t)outline.m_rooms.size();

					outline.m_start = 0;
					outline.m_finish = count - 1;

					count += extra_count;

					while (outline.m_rooms.size() < count)
					{
						// don't attach to last room
						int32_t last_room{ wind::random::generate(current_size - 2) };

						attach_room(outline, outline.m_rooms[last_room], count - 2);
					}

					clean_up(outline);

					return 0;
				}
#ifdef _DEBUG
				auto draw(const door_t& door, const ALLEGRO::POINT<int32_t>& position) -> void
				{
					int32_t r{ wind::random::generate(64) + 128 };
					int32_t g{ wind::random::generate(64) + 128 };
					int32_t b{ wind::random::generate(64) + 128 };

					al::draw_circle({ door.x + position.x, door.y + position.y }, 3.0f, al::map_rgb(r, g, b), 1.0f);
				}

				auto draw(const room_t& room, const ALLEGRO::POINT<int32_t>& position) -> void
				{
					int32_t r{ wind::random::generate(64) + 128 };
					int32_t g{ wind::random::generate(64) + 128 };
					int32_t b{ wind::random::generate(64) + 128 };

					ALLEGRO::POINT<float> pos1{ (float)(position.x + room.top_left.x), (float)(position.y + room.top_left.y) };
					ALLEGRO::POINT<float> pos2{ (float)(position.x + room.bottom_right.x), (float)(position.y + room.bottom_right.y) };

					al::draw_rectangle(pos1, pos2, al::map_rgb(r, g, b), 1.0f);
				}

				auto draw(const outline_t& outline, const ALLEGRO::POINT<int32_t>& position) -> void
				{
					for (auto it = outline.m_rooms.cbegin(); it != outline.m_rooms.cend(); ++it)
					{
						draw(*it, position);
					}

					for (auto it = outline.m_doors.cbegin(); it != outline.m_doors.cend(); ++it)
					{
						draw(*it, position);
					}
				}
#endif
			}
		}
	}
}