module game;

import <cstdint>;
import <memory>;
import allegro;
import wind;
import :base;
import :dungeon.base;
import :dungeon.map.cell;
import :dungeon.map;
import :dungeon.map.outline;
import :dungeon.map.generator;

namespace game
{
	namespace dungeon
	{
		namespace map
		{
			generator_t::generator_t() : m_map(nullptr), m_outline(), m_bottom(false)
			{
			}

			generator_t::~generator_t()
			{
			}

			auto generator_t::generate(map_t& map, bool is_bottom_floor) -> int32_t
			{
				generator_t generator{};

				generator.m_map = &map;
				generator.m_bottom = is_bottom_floor;

				return generator.run();
			}

			auto generator_t::run() -> int32_t
			{
				if (outline::generate(this->m_outline, GAME::DUNGEON::GENERATOR::DIRECT_ROOMS, GAME::DUNGEON::GENERATOR::EXTRA_ROOMS) < 0)
				{
					return -1;
				}

				this->m_map->reset(this->m_outline.m_size);

				if (!this->m_map->m_data)
				{
					return -1;
				}

				if (this->populate_map() < 0)
				{
					return -1;
				}

				for (size_t i = 0; i < this->m_outline.m_rooms.size(); ++i)
				{
					this->populate_room(this->m_outline.m_rooms[i]);
				}

				return 0;
			}

			auto generator_t::is_door_to_room(const outline::room_t& room, const outline::door_t& door) -> bool
			{
				return ((door.x == room.top_left.x && door.y >= room.top_left.y && door.y <= room.bottom_right.y) ||
					(door.x == room.bottom_right.x && door.y >= room.top_left.y && door.y <= room.bottom_right.y) ||
					(door.y == room.top_left.y && door.x >= room.top_left.x && door.x <= room.bottom_right.x) ||
					(door.y == room.bottom_right.y && door.x >= room.top_left.x && door.x <= room.bottom_right.x));
			}

			auto generator_t::find_door_rooms(const outline::door_t& door, size_t(&rooms)[2]) -> void
			{
				size_t index{ 0 };
				size_t room_count{ this->m_outline.m_rooms.size() };

				for (size_t i = 0; i < room_count; ++i)
				{
					outline::room_t& room = this->m_outline.m_rooms[i];

					if (is_door_to_room(room, door))
					{
						rooms[index] = i;
						++index;
					}
				}
			}

			auto generator_t::find_room_center(const outline::room_t& room) -> ALLEGRO::POINT<int32_t>
			{
				return ALLEGRO::POINT<int32_t>{ (room.bottom_right.x + room.top_left.x) >> 1, (room.bottom_right.y + room.top_left.y) >> 1 };
			}

			auto generator_t::populate_map() -> int32_t
			{
				size_t room_count = this->m_outline.m_rooms.size();
				size_t door_count = this->m_outline.m_doors.size();

				for (size_t i = 0; i < room_count; ++i)
				{
					ALLEGRO::POINT<int32_t> pos{ 0,0 };

					outline::room_t& room = this->m_outline.m_rooms[i];

					for (pos.y = room.top_left.y; pos.y <= room.bottom_right.y; ++pos.y)
					{
						for (pos.x = room.top_left.x; pos.x <= room.bottom_right.x; ++pos.x)
						{
							cell_t& cell = this->m_map->at(pos.x + pos.y * this->m_map->m_size.width);

							cell.m_data.m_visible = 0;
							cell.m_data.m_floor = GAME::DUNGEON::CELL::TYPE::FLOOR_BRICK;
							cell.m_data.m_wall = GAME::DUNGEON::CELL::TYPE::BLANK;
							cell.m_data.m_object = GAME::DUNGEON::CELL::TYPE::BLANK;
							cell.m_data.m_edge = 0;
							cell.m_data.m_blocked = 0;
							cell.m_data.m_npc = 0;
							cell.m_data.m_trigger = 0;
							cell.m_data.m_fluff = 0;

							if (pos.x == room.top_left.x || pos.x == room.bottom_right.x ||
								pos.y == room.top_left.y || pos.y == room.bottom_right.y)
							{
								cell.m_data.m_edge = 1;
								cell.m_data.m_wall = GAME::DUNGEON::CELL::TYPE::WALL_BRICK;
								cell.m_data.m_blocked = 1;
							}
						}
					}
				}

				for (size_t i = 0; i < door_count; ++i)
				{
					size_t rooms[2]{ 0, 0 };

					outline::door_t& door = this->m_outline.m_doors[i];
					this->find_door_rooms(door, rooms);

					cell_t& cell = this->m_map->at(door.x + door.y * this->m_map->m_size.width);

					if (rooms[0] == this->m_outline.m_finish || rooms[1] == this->m_outline.m_finish)
					{
						this->m_map->m_boss = door;

						if (this->m_bottom)
						{
							cell.m_data.m_wall = GAME::DUNGEON::CELL::TYPE::DOOR_BRICK_BOSS_CLOSED;
						}
						else
						{
							cell.m_data.m_wall = GAME::DUNGEON::CELL::TYPE::DOOR_BRICK_SEALED_CLOSED;
						}
					}
					else
					{
						cell.m_data.m_wall = GAME::DUNGEON::CELL::TYPE::DOOR_BRICK_STANDARD_CLOSED;
					}
				}

				for (auto it = this->m_outline.m_rooms.cbegin(); it != this->m_outline.m_rooms.cend(); ++it)
				{
					if (populate_room((*it)) < 0)
					{
						return -1;
					}
				}

				return 0;
			}

			auto generator_t::populate_room(const outline::room_t& room) -> int32_t
			{
				if (&room == &this->m_outline.m_rooms[this->m_outline.m_start])
				{
					cell_t& cstart = this->m_map->at(this->m_map->m_start.x + this->m_map->m_start.y * this->m_map->m_size.width);
					cstart.m_data.m_blocked = 0;
					cstart.m_data.m_object = GAME::DUNGEON::CELL::TYPE::STAIR_UP;
					cstart.m_data.m_wall = GAME::DUNGEON::CELL::TYPE::BLANK;
					cstart.m_data.m_trigger = 1;

					this->m_map->m_start = this->find_room_center(room);
					return 0;
				}

				if (&room == &this->m_outline.m_rooms[this->m_outline.m_finish])
				{
					cell_t& cfinish = this->m_map->at(this->m_map->m_finish.x + this->m_map->m_finish.y * this->m_map->m_size.width);
					if (!this->m_bottom)
					{
						cfinish.m_data.m_blocked = 0;
						cfinish.m_data.m_object = GAME::DUNGEON::CELL::TYPE::STAIR_DOWN;
						cfinish.m_data.m_wall = GAME::DUNGEON::CELL::TYPE::BLANK;
						cfinish.m_data.m_trigger = 1;
					}
					this->m_map->m_finish = this->find_room_center(room);
					return 0;
				}

				return 0;
			}
		}
	}
}