module game;

import <cstdint>;
import <memory>;
import <map>;
import allegro;
import wind;
import :base;
import :dungeon.base;
import :dungeon.map;
import :dungeon.map.generator;

namespace game
{
	namespace dungeon
	{
		map_t::map_t() : m_data(), m_size({ 0, 0 }), m_start({ 0, 0 }), m_finish({ 0, 0 }), m_boss({ 0, 0 }) {}

		map_t::map_t(const map_t& map) : m_data(map.m_data), m_size(map.m_size), m_start(map.m_start), m_finish(map.m_finish), m_boss(m_boss) {}

		map_t::~map_t() {}

		auto map_t::operator = (const map_t& map) -> map_t&
		{
			this->m_data = map.m_data;

			return *this;
		}

		map_t::operator bool() const
		{
			return (bool)this->m_data;
		}

		auto map_t::clear() -> void
		{
			this->m_data.reset();
			this->m_size = { 0, 0 };
			this->m_start = { 0, 0 };
			this->m_finish = { 0, 0 };
			this->m_boss = { 0, 0 };
		}

		auto map_t::reset(size_t width, size_t height) -> bool
		{
			this->clear();

			this->m_data = std::make_shared<map_t::element_type[]>(width * height);
			if (this->m_data)
			{
				this->m_size = ALLEGRO::SIZE(width, height);
			}

			return (bool)this->m_data;
		}

		auto map_t::reset(const ALLEGRO::SIZE<size_t>& size) -> bool
		{
			return this->reset(size.width, size.height);
		}

		auto map_t::data() -> void*
		{
			return this->m_data.get();
		}

		auto map_t::data() const -> const void*
		{
			return this->m_data.get();
		}

		auto map_t::size() const -> const ALLEGRO::SIZE<size_t>&
		{
			return this->m_size;
		}

		auto map_t::at(size_t index) -> map_t::reference_element_type
		{
			ALLEGRO::ASSERT(index < (this->m_size.width * this->m_size.height));
			return this->m_data.get()[index];
		}

		auto map_t::at(size_t index) const -> map_t::const_reference_element_type
		{
			ALLEGRO::ASSERT(index < (this->m_size.width * this->m_size.height));
			return this->m_data.get()[index];
		}

		auto map_t::operator [](size_t index) -> map_t::reference_element_type
		{
			ALLEGRO::ASSERT(index < (this->m_size.width * this->m_size.height));
			return this->m_data.get()[index];
		}

		auto map_t::operator [](size_t index) const -> map_t::const_reference_element_type
		{
			ALLEGRO::ASSERT(index < (this->m_size.width * this->m_size.height));
			return this->m_data.get()[index];
		}

		auto map_t::begin() -> map_t::iterator
		{
			return iterator(this->m_data, 0);
		}

		auto map_t::end() -> map_t::iterator
		{
			return iterator(this->m_data, this->m_size.width * this->m_size.height);
		}

		auto map_t::cbegin() const -> map_t::const_iterator
		{
			return const_iterator(this->m_data, 0);
		}

		auto map_t::cend() const -> map_t::const_iterator
		{
			return const_iterator(this->m_data, this->m_size.width * this->m_size.height);
		}

		auto map_t::get_start() const->const ALLEGRO::POINT<int32_t>&
		{
			return this->m_start;
		}

		auto map_t::get_finish() const->const ALLEGRO::POINT<int32_t>&
		{
			return this->m_finish;
		}

		auto map_t::get_boss_door() const->const ALLEGRO::POINT<int32_t>&
		{
			return this->m_boss;
		}

		auto map_t::unlock_room(const ALLEGRO::POINT<int32_t>& point) -> void
		{
			ALLEGRO::POINT<int32_t> top_left{ point };
			ALLEGRO::POINT<int32_t> bottom_right{ point };

			do
			{
				--top_left.x;
			} while (!this->m_data.get()[top_left.x + point.y * this->m_size.width].m_data.m_edge);

			do
			{
				++bottom_right.x;
			} while (!this->m_data.get()[bottom_right.x + point.y * this->m_size.width].m_data.m_edge);

			do
			{
				--top_left.y;
			} while (!this->m_data.get()[point.x + top_left.y * this->m_size.width].m_data.m_edge);

			do
			{
				++bottom_right.y;
			} while (!this->m_data.get()[point.x + bottom_right.y * this->m_size.width].m_data.m_edge);

			for (size_t j = top_left.y; j <= bottom_right.y; ++j)
			{
				for (size_t i = top_left.x; i <= bottom_right.x; ++i)
				{
					this->m_data.get()[i + j * this->m_size.width].m_data.m_visible = 1;
				}
			}
		}

		auto map_t::generate_new(bool is_bottom_floor) -> int32_t
		{
			return map::generator_t::generate(*this, is_bottom_floor);
		}

		auto map_t::get_index(const ALLEGRO::POINT<int32_t>& point) const -> int32_t
		{
			if (point.x < 0 || point.x >= this->m_size.width ||
				point.y < 0 || point.y >= this->m_size.height)
			{
				return -1;
			}

			return point.x + point.y * (int32_t)this->m_size.width;
		}

		auto map_t::get_cell(const ALLEGRO::POINT<int32_t>& point, GAME::DIRECTION direction) -> map::cell_t&
		{
			ALLEGRO::POINT<int32_t> pos{ point + dungeon::get_delta(direction) };

			size_t index{ pos.x + pos.y * this->m_size.width };

			return this->at(index);
		}
		auto map_t::get_cell(const ALLEGRO::POINT<int32_t>& point, GAME::DIRECTION direction) const-> const map::cell_t&
		{
			ALLEGRO::POINT<int32_t> pos{ point + dungeon::get_delta(direction) };

			size_t index{ pos.x + pos.y * this->m_size.width };

			return this->at(index);
		}

		auto map_t::get_trigger_map() const -> const trigger_map_t&
		{
			return this->m_trigger_map;
		}

		auto map_t::get_trigger_map() -> trigger_map_t&
		{
			return this->m_trigger_map;
		}
	}
}