module game;

import <cstdint>;
import <vector>;
import <memory>;
import <map>;
import allegro;
import wind;
import :base;
import :dungeon.base;
import :dungeon.dice;
import :dungeon.item.base;
import :dungeon.item.weapon;
import :dungeon.item.parser;

namespace GAME
{
	namespace DUNGEON
	{
		namespace ITEM
		{
			namespace WEAPON
			{
				enum class ATTRIBUTE_KEY : int32_t
				{
					INVALID = -1,
					NAME,
					TYPE,
					COST,
					WEIGHT,
					DAMAGE,
					PROPERTY,
					COUNT
				};
			}
		}
	}
}

namespace game
{
	namespace dungeon
	{
		namespace item
		{
			weapon_t::weapon_t()
			{
				this->m_name = "[WEAPON]";
			}

			weapon_t::~weapon_t()
			{
			}

			auto weapon_t::clone() const->item_t
			{
				std::shared_ptr<weapon_t> rv = std::make_shared<weapon_t>();
				
				if (rv)
				{
					std::memcpy((void*)rv.get(), (void*)this, sizeof(weapon_t));
				}

				return rv;
			}

			auto weapon_t::get_damage() const->const weapon::damage_t&
			{
				return this->m_damage;
			}

			auto weapon_t::set_damage(const weapon::damage_t& damage) -> void
			{
				this->m_damage.first = damage.first;
				this->m_damage.second.m_count = damage.second.m_count;
				this->m_damage.second.m_sides = damage.second.m_sides;
			}

			auto weapon_t::get_versatile() const->const weapon::damage_t&
			{
				return this->m_damage;
			}

			auto weapon_t::set_versatile(const weapon::damage_t& damage) -> void
			{
				this->m_damage.first = damage.first;
				this->m_damage.second.m_count = damage.second.m_count;
				this->m_damage.second.m_sides = damage.second.m_sides;
			}

			auto weapon_t::get_property() const->int32_t
			{
				return this->m_property;
			}

			auto weapon_t::set_property(int32_t property) -> void
			{
				this->m_property = property;
			}

			namespace weapon
			{
				std::map<wind::string_t, int32_t> m_icons{};

				const std::map<GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY, const wind::string_t> m_attributes
				{{
					{ GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY::NAME, "name" },
					{ GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY::TYPE, "type" },
					{ GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY::COST, "cost" },
					{ GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY::WEIGHT, "weight" },
					{ GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY::DAMAGE, "damage" },
					{ GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY::PROPERTY, "properties" }
				}};

				const std::map<wind::string_t, int32_t> m_properties
				{{
					{ "light", GAME::DUNGEON::ITEM::WEAPON::PROPERTY::LIGHT },
					{ "finesse", GAME::DUNGEON::ITEM::WEAPON::PROPERTY::FINESSE },
					{ "heavy", GAME::DUNGEON::ITEM::WEAPON::PROPERTY::HEAVY },
					{ "thrown", GAME::DUNGEON::ITEM::WEAPON::PROPERTY::THROWN },
					{ "two_handed", GAME::DUNGEON::ITEM::WEAPON::PROPERTY::TWO_HANDED },
					{ "versatile", GAME::DUNGEON::ITEM::WEAPON::PROPERTY::VERSATILE }
				}};

				const std::map<wind::string_t, int32_t> m_damages
				{ {
					{ "bludgeoning", GAME::DUNGEON::ITEM::WEAPON::DAMAGE_TYPE::BLUDGEONING },
					{ "slashing", GAME::DUNGEON::ITEM::WEAPON::DAMAGE_TYPE::SLASHING },
					{ "piercing", GAME::DUNGEON::ITEM::WEAPON::DAMAGE_TYPE::PIERCING }
				} };

				auto parse_die(const wind::string_t& string, int32_t& count, int32_t& sides) -> int32_t
				{
					std::vector<wind::string_t> vector{};
					size_t size = wind::string::separate(string, vector, 'd');
					if (size != 2)
					{
						return -1;
					}

					for (auto c : vector[0])
					{
						if (c < '0' || c > '9')
						{
							return -1;
						}
					}

					for (auto s : vector[1])
					{
						if (s < '0' || s > '9')
						{
							return -1;
						}
					}

					count = vector[0].get_as<int32_t>();
					sides = vector[1].get_as<int32_t>();

					return 0;
				}

				auto parse_damage(const wind::string_t& string, weapon::damage_t& damage) -> int32_t
				{
					std::vector<wind::string_t> vector{};

					damage = weapon::damage_t{ -1, {0, 0} };

					size_t size = wind::string::separate(string, vector, ':');
					if (size < 2)
					{
						return -1;
					}

					auto it = m_damages.find(vector[0]);
					if (it == m_damages.cend())
					{
						return -1;
					}
					damage.first = it->second;

					if (parse_die(vector[1], damage.second.m_count, damage.second.m_sides) < 0)
					{
						return -1;
					}

					return 0;
				}

				auto reset() -> void
				{
					m_icons.clear();
				}

				auto parser(const wind::json_object_t& json) -> item_t
				{
					std::shared_ptr<weapon_t> item = std::make_shared<weapon_t>();

					wind::array_t<wind::string_t> values(std::to_underlying(GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY::COUNT));

					for (auto mit = m_attributes.cbegin(); mit != m_attributes.cend(); ++mit)
					{
						int32_t index = std::to_underlying(mit->first);
						const wind::string_t& name = mit->second;

						auto jit = json.find(name);
						if (jit == json.cend())
						{
							return nullptr;
						}
						
						const wind::string_t& string = jit->get_as_string();
						values[index] = string;

					}

					item->set_name(values[std::to_underlying(GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY::NAME)]);
					item->set_type(GAME::DUNGEON::ITEM::TYPE::WEAPON);

					int32_t icon_index = dungeon::item::icon::get_index(GAME::DUNGEON::ITEM::TYPE::WEAPON, values[std::to_underlying(GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY::NAME)]);
					if (icon_index < 0)
					{
						return nullptr;
					}
					item->set_icon(icon_index);

					int32_t cost = item::parse_currency(values[std::to_underlying(GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY::COST)]);
					if (cost < 0)
					{
						return nullptr;
					}

					item->set_cost(cost);
					item->set_weight(values[std::to_underlying(GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY::WEIGHT)].get_as<float>());

					damage_t damage{};
					if (parse_damage(values[std::to_underlying(GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY::DAMAGE)], damage) < 0)
					{
						return nullptr;
					}

					item->set_damage(damage);
					//{ GAME::DUNGEON::ITEM::WEAPON::ATTRIBUTE_KEY::PROPERTY, "properties" }

					return item;
				}

				auto parse_icons(const wind::json_array_t& array) -> int32_t
				{
					return 0;
				}
			}
		}
	}
}