module game;

import <cstdint>;
import <vector>;
import <memory>;
import <map>;
import allegro;
import allegro.physfs_addon;
import wind;
import :base;
import :dungeon.base;
import :dungeon.item.base;
import :dungeon.item.list;
import :dungeon.item.parser;

namespace game
{
	namespace dungeon
	{
		namespace item
		{
			using map_t = typename std::map<wind::string_t, parser_func_t>;

			map_t m_map{};

			auto register_parser(const wind::string_t& type, parser_func_t parser) -> void
			{
				if (parser)
				{
					m_map[type] = parser;
				}
				else
				{
					if (m_map.find(type) != m_map.end())
					{
						m_map.erase(type);
					}
				}
			}

			auto parse_file(const wind::string_t& filename, list_t& list) -> int32_t
			{
				wind::json_t json{};

				if (!al::filename_exists(filename.c_str()))
				{
					return -1;
				}

				if (wind::json::load(json, filename) < 0)
				{
					return -1;
				}

				return item::parse_json(json, list);
			}

			auto parse_file_from_archive(const wind::string_t& archive_filename, const wind::string_t& filename, list_t& list) -> int32_t
			{
				wind::string_t filepath{ wind::path::make_canonical(archive_filename) };
				wind::string_t base{};
				wind::string_t ext{};
				wind::string_t path{};
				wind::json_t json{};
				bool archive = false;
				bool error = false;
				int32_t rv{ -1 };

				wind::path::split_filepath(filepath, path, base, ext);

				const PHYSFS_ArchiveInfo** i = nullptr;
				for (i = PHYSFS_supportedArchiveTypes(); *i != nullptr; i++)
				{
					if (wind::string::to_upper(ext) == (*i)->extension)
					{
						archive = true;
						break;
					}
				}

				if (archive)
				{
					const ALLEGRO::FILE_INTERFACE file_interface = al::get_new_file_interface();

					if (PHYSFS_mount(filename.c_str(), nullptr, 1))
					{
						al::physfs_addon::set_file_interface();
						rv = item::parse_file(filename, list);
						PHYSFS_unmount(filename.c_str());
					}

					al::set_new_file_interface(file_interface);
				}

				return rv;
			}

			auto parse_json(const wind::json_t& json, list_t& list) -> int32_t
			{
				if (m_map.size() == 0)
				{
					return -1;
				}

				if (json.get_type() != WIND::JSON::TYPE_OBJECT)
				{
					wind::lout << "\n\n********************\n" <<
						"File:" << __FILE__ <<
						"\nLine #:" << __LINE__ <<
						"\nError: Invalid JSON type" <<
						"\n********************\n" << wind::endl;
					return -1;
				}

				const wind::json_object_t& object = json.get_as_object();

				if (object.size() == 0)
				{
					wind::lout << "\n\n********************\n" <<
						"File:" << __FILE__ <<
						"\nLine #:" << __LINE__ <<
						"\nError: Empty JSON object" <<
						"\n********************\n" << wind::endl;
					return -1;
				}

				wind::json_object_t::const_iterator name_it = object.find("name");

				if (name_it == object.cend())
				{
					wind::lout << "\n\n********************\n" <<
						"File:" << __FILE__ <<
						"\nLine #:" << __LINE__ <<
						"\nError: Missing \"name\" element" <<
						"\n********************\n" << wind::endl;
					return -1;
				}

				if (name_it->get_type() != WIND::JSON::TYPE_STRING)
				{
					wind::lout << "\n\n********************\n" <<
						"File:" << __FILE__ <<
						"\nLine #:" << __LINE__ <<
						"\nError: Invalid JSON type" <<
						"\n********************\n" << wind::endl;
					return -1;
				}

				wind::string_t name_s = name_it->get_as_string();

				wind::json_object_t::const_iterator type_it = object.find("type");

				if (type_it == object.cend())
				{
					wind::lout << "\n\n********************\n" <<
						"File:" << __FILE__ <<
						"\nLine #:" << __LINE__ <<
						"\nError: Missing \"type\" element" <<
						"\n********************\n" << wind::endl;
					return -1;
				}

				if (type_it->get_type() != WIND::JSON::TYPE_STRING)
				{
					wind::lout << "\n\n********************\n" <<
						"File:" << __FILE__ <<
						"\nLine #:" << __LINE__ <<
						"\nError: Invalid JSON type" <<
						"\n********************\n" << wind::endl;
					return -1;
				}

				wind::string_t type_s = type_it->get_as_string();

				std::vector<wind::string_t> vector{};
				wind::string::separate(type_s, vector, ':');
				const wind::string_t& type_name = vector[0];

				map_t::const_iterator map_it = m_map.find(type_name);

				if (map_it == m_map.cend())
				{
					wind::lout << "\n\n********************\n" <<
						"File:" << __FILE__ <<
						"\nLine #:" << __LINE__ <<
						"\nError: Invalid item type" <<
						"\n********************\n" << wind::endl;
					return -1;
				}

				item_t item_element = map_it->second(object);

				if (!item_element)
				{
					wind::lout << "\n\n********************\n" <<
						"File:" << __FILE__ <<
						"\nLine #:" << __LINE__ <<
						"\n********************\n" << wind::endl;
					return -1;
				}

				list.add(name_s, item_element);

				return 0;
			}

			std::map<GAME::DUNGEON::CURRENCY, int32_t> m_currency =
			{{
				{GAME::DUNGEON::CURRENCY::COPPER, 1},
				{GAME::DUNGEON::CURRENCY::SILVER, 10},
				{GAME::DUNGEON::CURRENCY::GOLD, 100},
				{GAME::DUNGEON::CURRENCY::PLATINUM, 1000}
			}};

			auto parse_currency(const wind::string_t& cost) -> int32_t
			{
				const std::array<char, std::to_underlying(GAME::DUNGEON::CURRENCY::COUNT)> list =
				{
					'c', 's', 'g', 'p'
				};
				int32_t total{ 0 };
				int32_t value{ 0 };
				const char* s = cost.c_str();
				int32_t denomination{ -1 };

				while (*s)
				{
					if (*s >= '0' && *s <= '9')
					{
						value = value * 10 + *s - '0';
					}
					else
					{
						denomination = -1;

						for (int32_t i = 0; i < std::to_underlying(GAME::DUNGEON::CURRENCY::COUNT); ++i)
						{
							if (*s == list[i])
							{
								denomination = i;
							}
						}

						if (denomination < 0)
						{
							return -1;
						}

						value *= m_currency[(GAME::DUNGEON::CURRENCY)denomination];
						total += value;
						value = 0;
					}

					++s;
				}

				return total;
			}

			auto convert_currency_to_string(int32_t amount) -> wind::string_t
			{
				wind::string_t out{};

				for (int32_t denomination = std::to_underlying(GAME::DUNGEON::CURRENCY::PLATINUM); denomination >= std::to_underlying(GAME::DUNGEON::CURRENCY::COPPER); --denomination)
				{
					int32_t value = m_currency[(GAME::DUNGEON::CURRENCY)denomination];
					int32_t a = amount / m_currency[(GAME::DUNGEON::CURRENCY)denomination];

					if (a > 0)
					{
						amount -= (a * m_currency[(GAME::DUNGEON::CURRENCY)denomination]);
						out += wind::string::to_string(a);
					}
				}

				return out;
			}
		}
	}
}