module game;

import <cstdint>;
import <vector>;
import <memory>;
import <map>;
import allegro;
import allegro.physfs_addon;
import wind;
import :base;
import :dungeon.base;
import :dungeon.item.base;
import :dungeon.item.list;
import :dungeon.item.parser;

namespace game
{
	namespace dungeon
	{
		namespace item
		{
			list_t::list_t() : m_data()
			{
			}

			list_t::list_t(const list_t& list) : m_data(list.m_data)
			{
			}

			list_t::~list_t()
			{
			}

			list_t& list_t::operator = (const list_t& list)
			{
				this->m_data = list.m_data;
				return *this;
			}

			auto list_t::clear() -> void
			{
				return this->m_data.clear();
			}

			auto list_t::size() const->size_t
			{
				return this->m_data.size();
			}

			auto list_t::add(const map_key_type& key, element_type& element) -> void
			{
				auto it = this->m_index.find(key);

				if (it == this->m_index.end())
				{
					this->m_index[key] = this->m_data.size();
					this->m_data.push_back(element);
				}
				else
				{
					this->m_data[it->second] = element;
				}
			}

			auto list_t::at(size_t index) -> reference_element_type
			{
				ALLEGRO::ASSERT(index < this->m_data.size());
				return this->m_data.at(index);
			}

			auto list_t::at(size_t index) const->const_reference_element_type
			{
				ALLEGRO::ASSERT(index < this->m_data.size());
				return this->m_data.at(index);
			}

			auto list_t::operator [](size_t index)->reference_element_type
			{
				ALLEGRO::ASSERT(index < this->m_data.size());
				return this->m_data.at(index);
			}

			auto list_t::operator [](size_t index) const->const_reference_element_type
			{
				ALLEGRO::ASSERT(index < this->m_data.size());
				return this->m_data.at(index);
			}

			auto list_t::exists(const map_key_type& name) const -> bool
			{
				auto it = this->m_index.find(name);

				if (it != this->m_index.cend())
				{
					return true;
				}

				return false;
			}

			auto list_t::find(const map_key_type& name) -> map_type::iterator
			{
				return this->m_index.find(name);
			}

			auto list_t::find(const map_key_type& name) const-> map_type::const_iterator
			{
				return this->m_index.find(name);
			}


			auto list_t::get_by_type(GAME::DUNGEON::ITEM::TYPE type, std::vector<pair_t>& vector) -> size_t
			{
				vector.clear();

				for (auto it = this->m_index.begin(); it != this->m_index.end(); ++it)
				{
					if (it->second >= 0)
					{
						reference_element_type data = this->m_data.at(it->second);

						if (data->get_type() == type)
						{
							list_t::pair_t pair
							{
								it->first,
								data
							};

							vector.push_back(pair);
						}
					}
				}

				return vector.size();
			}

			auto list_t::get_by_type(GAME::DUNGEON::ITEM::TYPE type, std::vector<const_pair_t>& vector) const->size_t
			{
				vector.clear();

				for (auto it = this->m_index.begin(); it != this->m_index.end(); ++it)
				{
					if (it->second >= 0)
					{
						const_reference_element_type data = this->m_data.at(it->second);

						if (data->get_type() == type)
						{
							list_t::const_pair_t pair
							{
								it->first,
								data
							};

							vector.push_back(pair);
						}
					}
				}

				return vector.size();
			}

			auto list_t::begin() -> iterator
			{
				return iterator(this->m_data, this->m_index, this->m_index.begin());
			}

			auto list_t::end() -> iterator
			{
				return iterator(this->m_data, this->m_index, this->m_index.end());
			}

			auto list_t::cbegin() const->const_iterator
			{
				return const_iterator(this->m_data, this->m_index, this->m_index.cbegin());
			}

			auto list_t::cend() const->const_iterator
			{
				return const_iterator(this->m_data, this->m_index, this->m_index.cend());
			}

			namespace list
			{
				auto parse_json_manifest(const wind::json_t& json, list_t& list) -> int32_t
				{
					if (json.get_type() != WIND::JSON::TYPE_OBJECT)
					{
						wind::lout << "\n\n********************\n" <<
							"File:" << (const char*)__FILE__ <<
							"\nLine #:" << wind::string::to_string(__LINE__) <<
							"\n********************\n" << wind::endl;
						return -1;
					}

					const wind::json_object_t& object = json.get_as_object();

					auto items_it = object.find("items");
					if (items_it == object.cend())
					{
						wind::lout << "\n\n********************\n" <<
							"File:" << __FILE__ <<
							"\nLine #:" << wind::string::to_string(__LINE__) <<
							"\n********************\n" << wind::endl;
						return -1;
					}

					if (items_it->get_type() != WIND::JSON::TYPE_ARRAY)
					{
						wind::lout << "\n\n********************\n" <<
							"File:" << __FILE__ <<
							"\nLine #:" << wind::string::to_string(__LINE__) <<
							"\n********************\n" << wind::endl;
						return -1;
					}

					const wind::json_array_t& item_array = items_it->get_as_array();

					for (auto it = item_array.cbegin(); it != item_array.cend(); ++it)
					{
						if (it->get_type() != WIND::JSON::TYPE_OBJECT)
						{
							wind::lout << "\n\n********************\n" <<
								"File:" << __FILE__ <<
								"\nLine #:" << wind::string::to_string(__LINE__) <<
								"\n********************\n" << wind::endl;
							return -1;
						}

						const wind::json_object_t& item_object = it->get_as_object();
						
						if (dungeon::item::parse_json(item_object, list) < 0)
						{
							wind::lout << "\n\n********************\n" <<
								"File:" << __FILE__ <<
								"\nLine #:" << wind::string::to_string(__LINE__) <<
								"\n********************\n" << wind::endl;
							return -1;
						}
					}

					return 0;
				}

				auto parse_json(const wind::json_t& json, list_t& list) -> int32_t
				{
					if (json.get_type() != WIND::JSON::TYPE_OBJECT)
					{
						wind::lout << "\n\n********************\n" <<
							"File:" << __FILE__ <<
							"\nLine #:" << wind::string::to_string(__LINE__) <<
							"\n********************\n" << wind::endl;
					}

					const wind::json_object_t& object = json.get_as_object();

					auto file_it = object.find("files");
					if (file_it == object.cend())
					{
						wind::lout << "\n\n********************\n" <<
							"File:" << __FILE__ <<
							"\nLine #:" << wind::string::to_string(__LINE__) <<
							"\n********************\n" << wind::endl;
						return -1;
					}

					if (file_it->get_type() != WIND::JSON::TYPE_ARRAY)
					{
						wind::lout << "\n\n********************\n" <<
							"File:" << __FILE__ <<
							"\nLine #:" << wind::string::to_string(__LINE__) <<
							"\n********************\n" << wind::endl;
						return -1;
					}

					const wind::json_array_t& file_array = file_it->get_as_array();

					for (auto it = file_array.cbegin(); it != file_array.cend(); ++it)
					{
						wind::json_t json_manifest{};

						if (it->get_type() != WIND::JSON::TYPE_STRING)
						{
							return -1;
							wind::lout << "\n\n********************\n" <<
								"File:" << __FILE__ <<
								"\nLine #:" << wind::string::to_string(__LINE__) <<
								"\n********************\n" << wind::endl;
							return -1;
						}

						const wind::string_t& filename = it->get_as_string();

						if (wind::json::load(json_manifest, wind::string_t(GAME::MANIFESTS_DIRECTORY) + "\\" + filename) < 0)
						{
							wind::lout << "\n\n********************\n" <<
								"File:" << __FILE__ <<
								"\nLine #:" << wind::string::to_string(__LINE__) <<
								"\n********************\n" << wind::endl;
							return -1;
						}

						if (list::parse_json_manifest(json_manifest, list) < 0)
						{
							wind::lout << "\n\n********************\n" <<
								"File:" << __FILE__ <<
								"\nLine #:" << wind::string::to_string(__LINE__) <<
								"\n********************\n" << wind::endl;
							return -1;
						}
					}

					return 0;
				}

				auto load_items(const wind::string_t& filename, list_t& list) -> int32_t
				{
					wind::json_t json{};

					dungeon::item::register_parser("armor", dungeon::item::armor::parser);
					dungeon::item::register_parser("food", dungeon::item::food::parser);
					dungeon::item::register_parser("potion", dungeon::item::potion::parser);
					dungeon::item::register_parser("treasure", dungeon::item::treasure::parser);
					dungeon::item::register_parser("weapon", dungeon::item::weapon::parser);


					wind::lout << "\nDoes file exist: ";
					if (!al::filename_exists(filename.c_str()))
					{
						wind::lout << "failed" << wind::endl;
						return -1;
					}

					wind::lout << "\nLoad JSON file: ";
					if (wind::json::load(json, filename) < 0)
					{
						wind::lout << "failed" << wind::endl;
						return -1;
					}

					return list::parse_json(json, list);
				}

				auto load_items_from_archive(const wind::string_t& archive_filename, const wind::string_t& filename, list_t& list) -> int32_t
				{
					wind::string_t filepath{ wind::path::make_canonical(archive_filename) };
					wind::string_t base{};
					wind::string_t ext{};
					wind::string_t path{};
					wind::json_t json{};
					bool archive = false;
					bool error = false;
					int32_t rv{ -1 };

					wind::path::split_filepath(filepath, path, base, ext);

					const PHYSFS_ArchiveInfo** i = nullptr;
					for (i = PHYSFS_supportedArchiveTypes(); *i != nullptr; i++)
					{
						if (wind::string::to_upper(ext) == (*i)->extension)
						{
							archive = true;
							break;
						}
					}

					if (archive)
					{
						const ALLEGRO::FILE_INTERFACE file_interface = al::get_new_file_interface();

						if (PHYSFS_mount(filename.c_str(), nullptr, 1))
						{
							al::physfs_addon::set_file_interface();
							rv = list::load_items(filename, list);
							PHYSFS_unmount(filename.c_str());
						}

						al::set_new_file_interface(file_interface);
					}

					return rv;
				}
			}
		}
	}
}