module game;

import <cstdint>;
import <memory>;
import allegro;
import wind;
import wind.datafile_addon;
import allegro.primitives_addon;
import :base;
import :game_shared_data;
import :process;
import :dungeon.base;
import :dungeon.map.cell;
import :dungeon.map.camera;
import :dungeon.trigger;
import :dungeon.dungeon_shared_data;

namespace game
{
	dungeon_t::dungeon_t()
	{
		this->m_process = GAME::PROCESS::DUNGEON;
	}

	dungeon_t::~dungeon_t()
	{
	}

	auto dungeon_t::on_initialize() -> int32_t
	{
		this->m_state = GAME::DUNGEON::STATE::INITIALIZING;

		wind::lout << "Generating Map: ";
		if (this->m_map.generate_new() < 0)
		{
			wind::lout << "failed" << wind::endl;
			return -1;
		}
		wind::lout << "pass" << wind::endl;

		const ALLEGRO::POINT<int32_t>& finish{ this->m_map.get_finish() };
		this->m_map.unlock_room(finish);
		this->m_player.set_bitmap(std::static_pointer_cast<ALLEGRO::BITMAP_DATA>(game_shared_data_t::get_datafile()[GAME::DATAFILE::SHARED_SARA]));
		this->m_player.set_position({ finish.x, finish.y });
		this->m_player.set_facing(GAME::DIRECTION::SOUTH);
		this->m_camera.set_position({ (finish.x - (int32_t)(GAME::DUNGEON::CAMERA::VIEWPORT.width >> 1)), (finish.y - (int32_t)(GAME::DUNGEON::CAMERA::VIEWPORT.height >> 1)) });
		this->m_camera.follow(this->m_player.get_position());

		int32_t index{ this->m_map.get_index(this->m_map.get_finish()) };
		this->m_map[index].set_object_tile(GAME::DUNGEON::CELL::TYPE::BLANK);
		this->m_map[index].set_trigger(0);
		int32_t bones{ 0 };

		do
		{
			ALLEGRO::POINT<int32_t> skeleton{ finish.x + wind::random::generate(9) - 4, finish.y + wind::random::generate(7) - 3 };
			bones = this->m_map.get_index(skeleton);
		} while (bones == index);

		this->m_map[bones].set_object_tile(GAME::DUNGEON::CELL::TYPE::SKELETON);
		this->m_map[bones].set_trigger(1);

		dungeon::trigger_t trigger = std::make_shared<dungeon::trigger::skeleton_t>();
		if (!trigger)
		{
			return -1;
		}
		
		trigger->set_index(bones);
		trigger->set_type(GAME::DUNGEON::TRIGGER::TYPE::SKELETON);
		trigger->set_state(GAME::DUNGEON::TRIGGER::STATE::LIVE);
		trigger->set_flags(GAME::DUNGEON::TRIGGER::FLAG::VISIBLE | GAME::DUNGEON::TRIGGER::FLAG::STEP_INTO);

		dungeon::trigger_map_t& trigger_map = this->m_map.get_trigger_map();

		trigger_map[bones] = trigger;

		wind::lout << "Creating Local Data Structure: ";
		this->m_data = std::make_shared<dungeon::dungeon_shared_data_tag_t>();
		if (!this->m_data)
		{
			wind::lout << "failed" << wind::endl;
			return -1;
		}
		wind::lout << "pass" << wind::endl;
		game_shared_data_t::set_dungeon_data(this->m_data);

		wind::lout << "Loading Icon Data: ";
		if (dungeon::item::icon::load(wind::string_t(GAME::MANIFESTS_DIRECTORY) + "\\icons.json") < 0)
		{
			wind::lout << "failed" << wind::endl;
			return -1;
		}
		wind::lout << "pass" << wind::endl;

		wind::lout << "Loading Item Data: ";
		if (dungeon::item::list::load_items(wind::string_t(GAME::MANIFESTS_DIRECTORY) + "\\items.json", this->m_item_list) < 0)
		{
			wind::lout << "failed" << wind::endl;
			return -1;
		}
		wind::lout << "pass" << wind::endl;

		wind::lout << "Initializing Player: ";
		if (this->m_player.on_initialize() < 0)
		{
			wind::lout << "failed" << wind::endl;
			return -1;
		}
		wind::lout << "pass" << wind::endl;

		wind::lout << "Creating Inventory Screen: ";
		if (this->m_inventory.on_initialize() < 0)
		{
			wind::lout << "failed" << wind::endl;
			return -1;
		}
		wind::lout << "pass" << wind::endl;
		this->m_inventory.set_data(this->m_player.get_data());

		this->m_data->set_icon_tilesheet(GAME::DUNGEON::ITEM::TYPE::WEAPON,
			std::static_pointer_cast<wind::tilesheet_t>(game::game_shared_data_t::get_datafile()[GAME::DATAFILE::DUNGEON_TILESHEET_WEAPONS]));
	


		return 0;
	}

	auto dungeon_t::on_shutdown() -> int32_t
	{
		this->m_state = GAME::DUNGEON::STATE::CLOSING;
		this->m_inventory.on_shutdown();
		this->m_item_list.clear();
		this->m_map.clear();
		this->m_player.on_shutdown();
		game_shared_data_t::set_dungeon_data(nullptr);
		return 0;
	}

	auto dungeon_t::on_start() -> int32_t
	{
		this->m_state = GAME::DUNGEON::STATE::STARTING;
		wind::input::acknowledge();
		return 0;
	}

	auto dungeon_t::on_stop() -> int32_t
	{
		this->m_state = GAME::DUNGEON::STATE::STOPPING;
		return 0;
	}

	auto dungeon_t::on_update() -> GAME::STATE
	{
		static float movement_delay{ 0.0f };
		static GAME::DIRECTION move_direction = GAME::DIRECTION::INVALID;

		switch (this->m_state)
		{
		case GAME::DUNGEON::STATE::STARTING:
		{
			this->set_state(GAME::DUNGEON::STATE::WAITING);
		} break;
		case GAME::DUNGEON::STATE::WAITING:
		{
			wind::input::acknowledge();
			move_direction = GAME::DIRECTION::INVALID;
			this->set_state(GAME::DUNGEON::STATE::INPUT);
			movement_delay = 0.0f;
		} break;
		case GAME::DUNGEON::STATE::INPUT:
		{
			if (wind::input::keyboard::was_pressed(ALLEGRO::KEY_ESCAPE))
			{
				wind::input::keyboard::acknowledge(ALLEGRO::KEY_ESCAPE);
				return GAME::STATE::GAMEOVER;
			}

			if (wind::input::keyboard::is_pressed(ALLEGRO::KEY_E))
			{
				wind::input::keyboard::acknowledge(ALLEGRO::KEY_E);
				this->set_state(GAME::DUNGEON::STATE::INVENTORY);
				break;
			}

			if (movement_delay > 0.0f)
			{
				movement_delay -= GAME::DUNGEON::MOVEMENT::TICK;
			}
			else
			{
				movement_delay = 0.0f;

				if (wind::input::keyboard::is_pressed(ALLEGRO::KEY_RIGHT))
				{
					move_direction = GAME::DIRECTION::EAST;
					wind::input::keyboard::acknowledge(ALLEGRO::KEY_RIGHT);
				}

				if (wind::input::keyboard::is_pressed(ALLEGRO::KEY_LEFT))
				{
					move_direction = GAME::DIRECTION::WEST;
					wind::input::keyboard::acknowledge(ALLEGRO::KEY_LEFT);
				}

				if (wind::input::keyboard::is_pressed(ALLEGRO::KEY_UP))
				{
					move_direction = GAME::DIRECTION::NORTH;
					wind::input::keyboard::acknowledge(ALLEGRO::KEY_UP);
				}

				if (wind::input::keyboard::is_pressed(ALLEGRO::KEY_DOWN))
				{
					move_direction = GAME::DIRECTION::SOUTH;
					wind::input::keyboard::acknowledge(ALLEGRO::KEY_DOWN);
				}

				this->m_state = GAME::DUNGEON::STATE::WALKING;
			}
		} break;

		case GAME::DUNGEON::STATE::WALKING:
		{
			if (move_direction != GAME::DIRECTION::INVALID)
			{
				ALLEGRO::POINT<int32_t> new_point{ (ALLEGRO::POINT<int32_t>)this->m_player.get_position() + dungeon::get_delta(move_direction) };

				int32_t index = this->m_map.get_index(new_point);
				dungeon::map::cell_t& cell = this->m_map[index];

				if (cell.get_trigger())
				{
					if (this->process_trigger(index) >= 0)
					{
						break;
					}
				}

				if (cell.get_blocked())
				{
					uint32_t wall = cell.get_wall_tile();

					switch (wall)
					{
					case GAME::DUNGEON::CELL::TYPE::DOOR_BRICK_STANDARD_CLOSED:
					case GAME::DUNGEON::CELL::TYPE::DOOR_BRICK_SEALED_CLOSED:
					case GAME::DUNGEON::CELL::TYPE::DOOR_BRICK_BOSS_CLOSED:
					{
						new_point += dungeon::get_delta(move_direction);

						switch (wall)
						{
						case GAME::DUNGEON::CELL::TYPE::DOOR_BRICK_STANDARD_CLOSED:
						{
							cell.set_wall_tile(wall + 1);
							cell.set_blocked(0);
							this->m_map.unlock_room(new_point);
						} break;
						case GAME::DUNGEON::CELL::TYPE::DOOR_BRICK_SEALED_CLOSED:
						case GAME::DUNGEON::CELL::TYPE::DOOR_BRICK_BOSS_CLOSED:
						{
							if (this->m_player.get_data()->get_key())
							{
								cell.set_wall_tile(wall + 1);
								cell.set_blocked(0);
								this->m_map.unlock_room(new_point);
							}
						} break;
						}
					} break;
					}
				}

				if (!cell.get_blocked())
				{
					this->m_player.set_moved(true);
					this->m_player.set_position(this->m_player.get_position() + (ALLEGRO::POINT<float>)dungeon::get_delta(move_direction));
					this->m_player.set_facing(move_direction);
					movement_delay = GAME::DUNGEON::MOVEMENT::DELAY;
					move_direction = GAME::DIRECTION::INVALID;
				}
			}

			this->m_player.on_update();
			this->set_state(GAME::DUNGEON::STATE::WAITING);
			wind::system::timer::reset();

			this->m_camera.update();
		} break;

		case GAME::DUNGEON::STATE::INVENTORY:
		{
			GAME::DUNGEON::STATE state = this->m_inventory.on_update();
			
			if (state != GAME::DUNGEON::STATE::INVENTORY)
			{
				this->set_state(state);
			}
		} break;


		case GAME::DUNGEON::STATE::TRIGGER:
		{
			this->m_trigger->on_update();
			if (this->m_trigger->get_state() == GAME::DUNGEON::TRIGGER::STATE::DEACTIVATED)
			{
				if (this->m_trigger->get_type() == GAME::DUNGEON::TRIGGER::TYPE::SKELETON)
				{
					std::vector<dungeon::item::list_t::const_pair_t> weapons{};

					if (this->m_item_list.get_by_type(GAME::DUNGEON::ITEM::TYPE::WEAPON, weapons) > 0)
					{
						dungeon::item::list_t::const_pair_t& pair = weapons.at(19);

						dungeon::item_t& slot = this->m_player.get_data()->get_slot(0);
						slot = pair.second->clone();
					}
					this->m_player.get_data()->set_key(true);
				}

				this->m_trigger.reset();
				this->set_state(GAME::DUNGEON::STATE::WALKING);
			}
		} break;
		}

		return GAME::STATE::DUNGEON;
	}

	auto dungeon_t::on_render() -> void
	{
		if (this->m_state == GAME::DUNGEON::STATE::INVENTORY)
		{
			this->m_inventory.on_render();
		}
		else
		{
			const ALLEGRO::POINT<int32_t> position{ 0,0 };

			std::shared_ptr<wind::tilesheet_t> tilesheet = std::static_pointer_cast<wind::tilesheet_t>(game::game_shared_data_t::get_datafile()[GAME::DATAFILE::DUNGEON_TILESHEET_MAIN]);

			ALLEGRO::SIZE<int32_t> dim = al::get_display_dimensions(wind::system::display::get());

			al::clear_to_color(wind::map_rgba_i(0x000000ff));
			ALLEGRO::RECTANGLE clip = al::get_clipping_rectangle();
			al::set_clipping_rectangle({ position, {GAME::DUNGEON::CAMERA::VIEWPORT.width << GAME::DUNGEON::TILE::SHIFT.x, GAME::DUNGEON::CAMERA::VIEWPORT.height << GAME::DUNGEON::TILE::SHIFT.y} });

			this->m_map.draw(tilesheet, this->m_camera, position);
			this->m_player.on_render(this->m_camera, position);

			al::draw_filled_rectangle(position, { (float)(GAME::DUNGEON::CAMERA::VIEWPORT.width << GAME::DUNGEON::TILE::SHIFT.x), (float)(GAME::DUNGEON::CAMERA::VIEWPORT.height << GAME::DUNGEON::TILE::SHIFT.y) }, wind::map_rgba_i(0x00000000));

			al::set_clipping_rectangle(clip);

			if (this->m_state == GAME::DUNGEON::STATE::TRIGGER)
			{
				this->m_trigger->on_render();
			}
		}
	}

	auto dungeon_t::get_viewport() -> ALLEGRO::SIZE<int32_t>
	{
		if (this->m_state == GAME::DUNGEON::STATE::INVENTORY)
		{
			return this->m_inventory.get_viewport();
		}

		return { GAME::DUNGEON::CAMERA::VIEWPORT.width << GAME::DUNGEON::TILE::SHIFT.x, GAME::DUNGEON::CAMERA::VIEWPORT.height << GAME::DUNGEON::TILE::SHIFT.y };
	}

	auto dungeon_t::get_state() const->GAME::DUNGEON::STATE
	{
		return this->m_state;
	}

	auto dungeon_t::set_state(GAME::DUNGEON::STATE state) -> void
	{
		this->m_state = state;
		wind::input::acknowledge();
	}

	auto dungeon_t::get_map() -> dungeon::map_t&
	{
		return this->m_map;
	}

	auto dungeon_t::get_map() const -> const dungeon::map_t&
	{
		return this->m_map;
	}

	auto dungeon_t::get_player() -> dungeon::player_t&
	{
		return this->m_player;
	}

	auto dungeon_t::get_player() const -> const dungeon::player_t&
	{
		return this->m_player;
	}

	auto dungeon_t::get_camera() -> dungeon::map::camera_t&
	{
		return this->m_camera;
	}

	auto dungeon_t::get_camera() const -> const dungeon::map::camera_t&
	{
		return this->m_camera;
	}

	auto dungeon_t::ascend() -> int32_t
	{
		return 0;
	}

	auto dungeon_t::descend() -> int32_t
	{
		return 0;
	}

	auto dungeon_t::process_trigger(size_t index) -> int32_t
	{
		bool clear = false;
		dungeon::map::cell_t& cell = this->m_map[index];


		dungeon::trigger_map_t& trigger_map = this->m_map.get_trigger_map();

		auto it = trigger_map.find(index);
		if (it == trigger_map.end())
		{
			return -1;
		}

		dungeon::trigger_t& trigger = it->second;

		switch (trigger->get_type())
		{
		case GAME::DUNGEON::TRIGGER::TYPE::DOOR:
		{
		} break;
		case GAME::DUNGEON::TRIGGER::TYPE::CHEST:
		{
			cell.set_trigger(0);
			this->m_trigger = trigger;
			trigger_map.erase(index);
			this->set_state(GAME::DUNGEON::STATE::TRIGGER);
		} break;
		case GAME::DUNGEON::TRIGGER::TYPE::STAIR_DOWN:
		{
		} break;
		case GAME::DUNGEON::TRIGGER::TYPE::STAIR_UP:
		{
		} break;
		case GAME::DUNGEON::TRIGGER::TYPE::SKELETON:
		{
			cell.set_trigger(0);
			this->m_trigger = trigger;
			trigger_map.erase(index);
			this->set_state(GAME::DUNGEON::STATE::TRIGGER);
		} break;
		}

		return 0;
	}
}