module;

#define OUTPUT_BITMAP_03

module game;

import <cstdint>;
import <memory>;
import <cmath>;
import allegro;
import allegro.font_addon;
import allegro.ttf_addon;
import wind;
import wind.datafile_addon;
import :base;
import :game_shared_data;
import :process;
import :boardgame;
import :boardgame.base;
import :boardgame.mob;
import :boardgame.player;
import :boardgame.treasure;

namespace game
{
	namespace boardgame
	{
		auto build_background(ALLEGRO::BITMAP& background,
			std::array<boardgame::mob_t, (size_t)GAME::BOARDGAME::MOB::TYPE::COUNT>& mobs,
			std::array<boardgame::treasure_t, (size_t)GAME::BOARDGAME::TREASURE::TYPE::COUNT>& treasures,
			const wind::array_t<ALLEGRO::POINT<float>>& path) -> int32_t;
		auto build_path(wind::array_t<ALLEGRO::POINT<float>>& path) -> int32_t;

		namespace internal
		{
			static int32_t input_delay{ 0 };
			static int32_t walk_delay{ 0 };
			static int32_t attack_delay{ 0 };
			static int32_t recovery_delay{ 0 };
		}
	}

	boardgame_t::boardgame_t() : m_background()
	{
		this->m_process = GAME::PROCESS::BOARDGAME;
	}

	boardgame_t::~boardgame_t()
	{
	}

	auto boardgame_t::on_initialize() -> int32_t
	{
		this->m_state = GAME::BOARDGAME::STATE::INITIALIZING;

		this->m_caption = "Welcome!";

		if (boardgame::build_path(this->m_path) < 0)
		{
			return -1;
		}

		if (boardgame::build_background(this->m_background, this->m_mobs, this->m_treasures, this->m_path) < 0)
		{
			return -1;
		}

		ALLEGRO::BITMAP sara = std::static_pointer_cast<ALLEGRO::BITMAP_DATA>(game::game_shared_data_t::get_datafile()[GAME::DATAFILE::SHARED_SARA]);
		this->m_player.set_bitmap(sara);
		this->m_player.set_facing(1.5f * ALLEGRO::PI);
		this->m_player.set_frame(0);
		this->m_player.set_path(this->m_path);
		this->m_player.set_position(this->m_path[0]);
		this->m_player.set_state(GAME::BOARDGAME::PLAYER::STATE::IDLE);
		this->m_player.set_moves(0);
		this->m_player.set_dice(0);
		this->m_player.set_current_marker(0);
		this->m_player.set_next_marker(0);

		return 0;
	}

	auto boardgame_t::on_shutdown() -> int32_t
	{
		this->m_state = GAME::BOARDGAME::STATE::CLOSING;
		this->m_background.reset();
		this->m_path.clear();

		return 0;
	}

	auto boardgame_t::on_start() -> int32_t
	{
		this->m_state = GAME::BOARDGAME::STATE::STARTING;
		wind::input::acknowledge();
		return 0;
	}

	auto boardgame_t::on_stop() -> int32_t
	{
		this->m_state = GAME::BOARDGAME::STATE::STOPPING;
		return 0;
	}

	auto boardgame_t::on_update() -> GAME::STATE
	{
		switch (this->m_state)
		{
		case GAME::BOARDGAME::STATE::STARTING:
		{
			this->set_state(GAME::BOARDGAME::STATE::WAITING);
			this->m_caption = "Press ENTER to roll the dice!";
		} break;
		case GAME::BOARDGAME::STATE::WAITING:
		{
			--boardgame::internal::input_delay;
			if (boardgame::internal::input_delay <= 0)
			{
				this->m_caption = "Press ENTER to roll the dice!";
				this->set_state(GAME::BOARDGAME::STATE::INPUT);
				wind::input::keyboard::acknowledge(ALLEGRO::KEY_ENTER);
				boardgame::internal::walk_delay = wind::random::generate(20, 100);
			}
		} break;
		case GAME::BOARDGAME::STATE::INPUT:
		{
			if (wind::input::keyboard::was_pressed(ALLEGRO::KEY_ESCAPE))
			{
				wind::input::keyboard::acknowledge(ALLEGRO::KEY_ESCAPE);
				return GAME::STATE::GAMEOVER;
			}

			if (wind::input::keyboard::was_released(ALLEGRO::KEY_ENTER))
			{
				this->m_caption = "";
				this->set_state(GAME::BOARDGAME::STATE::ROLLING);
			}
			wind::input::acknowledge();
		}break;
		case GAME::BOARDGAME::STATE::ROLLING:
		{
			int32_t moves = 1 + wind::random::generate(GAME::BOARDGAME::DICE::COUNT);
			this->m_player.set_moves(moves);
			this->m_player.set_dice(moves);
			--boardgame::internal::walk_delay;
			if (boardgame::internal::walk_delay <= 0)
			{
				boardgame::internal::walk_delay = 0;
				this->set_state(GAME::BOARDGAME::STATE::WALKING);
			}
		}break;
		case GAME::BOARDGAME::STATE::WALKING:
		{
			this->m_player.on_update();
			wind::input::acknowledge();

			if (this->m_player.get_state() == GAME::BOARDGAME::PLAYER::STATE::STOPPED)
			{
				if (this->m_player.get_moves() == 0)
				{
					this->set_state(GAME::BOARDGAME::STATE::WAITING);
				}

				if (this->m_player.get_moves() == 0 || wind::random::generate(100) <= GAME::BOARDGAME::MOB::ATTACK_PERCENT)
				{
					for (auto& m : this->m_mobs)
					{
						if (m.get_marker() == this->m_player.get_current_marker())
						{
							this->m_caption = "Ouch!";
							this->m_player.set_state(GAME::BOARDGAME::PLAYER::STATE::HURTING);
							boardgame::internal::attack_delay = GAME::BOARDGAME::MOB::ATTACK_DELAY;
							this->set_state(GAME::BOARDGAME::STATE::ATTACKING);
						}
					}
				}

				for (auto& t : this->m_treasures)
				{
					if (t.get_marker() == this->m_player.get_current_marker())
					{
						if (t.get_state() != GAME::BOARDGAME::TREASURE::STATE::TAKEN)
						{
							this->m_caption = "Ooh, treasure!";
							t.set_state(GAME::BOARDGAME::TREASURE::STATE::TAKEN);
						}
					}
				}
			}
			else
			{
				if (this->m_player.get_state() == GAME::BOARDGAME::PLAYER::STATE::WON)
				{
					this->m_caption = "Ouch!";
					this->m_player.set_state(GAME::BOARDGAME::PLAYER::STATE::HURTING);
					boardgame::internal::attack_delay = GAME::BOARDGAME::MOB::ATTACK_DELAY;
					this->set_state(GAME::BOARDGAME::STATE::ATTACKING);

				}
			}
		}break;
		case GAME::BOARDGAME::STATE::ATTACKING:
		{
			if (boardgame::internal::attack_delay > 0)
			{
				--boardgame::internal::attack_delay;
			}
			else
			{
				this->m_caption = "You Lose!";
				this->set_state(GAME::BOARDGAME::STATE::RECOVERY);
				this->m_player.set_state(GAME::BOARDGAME::PLAYER::STATE::RECOVERY);
				boardgame::internal::recovery_delay = GAME::BOARDGAME::PLAYER::RECOVERY_DELAY;
				if (this->m_player.get_current_marker() == (this->m_path.get_count() - 1))
				{
					return GAME::STATE::DUNGEON;
				}
				this->m_caption = "Ooh! Try Again!";
			}
		}break;
		case GAME::BOARDGAME::STATE::RECOVERY:
		{
			if (boardgame::internal::recovery_delay > 0)
			{
				--boardgame::internal::recovery_delay;
			}
			else
			{
				this->set_state(GAME::BOARDGAME::STATE::WAITING);
				this->m_player.set_facing(1.5f * ALLEGRO::PI);
				this->m_player.set_frame(0);
				this->m_player.set_position(this->m_path[0]);
				this->m_player.set_state(GAME::BOARDGAME::PLAYER::STATE::IDLE);
				this->m_player.set_moves(0);
				this->m_player.set_dice(0);
				this->m_player.set_current_marker(0);
				this->m_player.set_next_marker(0);

			}
		} break;
		}

		for (auto& m : this->m_mobs)
		{
			m.on_update();
		}

		for (auto& t : this->m_treasures)
		{
			t.on_update();
		}

		return GAME::STATE::BOARDGAME;
	}

	auto boardgame_t::on_render() -> void
	{
		ALLEGRO::SIZE<float> bg_size{ (ALLEGRO::SIZE<float>)al::get_bitmap_dimensions(this->m_background) };
		al::draw_bitmap(this->m_background, { 0, 0 });

		for (auto& m : this->m_mobs)
		{
			m.on_render();
		}

		for (auto& t : this->m_treasures)
		{
			if (t.get_marker() == (this->m_path.get_count() - 1))
			{
				int32_t marker = this->m_player.get_current_marker();

				if (marker < (this->m_path.get_count() - 1))
				{
					t.on_render();
				}
				else
				{
					ALLEGRO::POINT<float> p1 = this->m_player.get_position();
					ALLEGRO::POINT<float> p2 = this->m_path.at(this->m_path.get_count() - 1);
					ALLEGRO::POINT<float> p3 = this->m_path.at(this->m_path.get_count() - 2);

					float r1 = wind::math::sqrt((p2.x - p1.x) * (p2.x - p1.x) + (p2.y - p1.y) * (p2.y - p1.y));
					float r2 = wind::math::sqrt((p2.x - p3.x) * (p2.x - p3.x) + (p2.y - p3.y) * (p2.y - p3.y));
					float r = r1 / r2;
					ALLEGRO::BITMAP k = std::static_pointer_cast<ALLEGRO::BITMAP_DATA>(game::game_shared_data_t::get_datafile()[GAME::DATAFILE::BOARDGAME_KRAMPUS]);
					ALLEGRO::BITMAP g = std::static_pointer_cast<ALLEGRO::BITMAP_DATA>(game::game_shared_data_t::get_datafile()[GAME::DATAFILE::BOARDGAME_PRESENT]);
					ALLEGRO::SIZE ks = al::get_bitmap_dimensions(k);
					ALLEGRO::SIZE gs = al::get_bitmap_dimensions(g);
					ALLEGRO::COLOR tint{1.0f, 1.0f, 1.0f, r};

					al::draw_tinted_bitmap(g, tint, { p2.x - (int32_t)(gs.width >> 1), p2.y - (int32_t)(gs.height >> 1) });
					tint.alpha = 1.0f - r;
					al::draw_tinted_bitmap(k, tint, { p2.x - (int32_t)(ks.width >> 1), p2.y - (int32_t)(ks.height >> 1) });
				}
			}
			else
			{
				t.on_render();
			}
		}
		
		ALLEGRO::BITMAP bg = std::static_pointer_cast<ALLEGRO::BITMAP_DATA>(game::game_shared_data_t::get_datafile()[GAME::DATAFILE::BOARDGAME_DICE]);
		ALLEGRO::RECTANGLE<int32_t> region{ {(this->m_player.get_dice()) * GAME::BOARDGAME::DICE::SIZE.width, 0}, GAME::BOARDGAME::DICE::SIZE };
		ALLEGRO::POINT<int32_t> position = GAME::BOARDGAME::DICE::FRAME::POSITION;

		position.x += ((GAME::BOARDGAME::DICE::FRAME::SIZE.width - GAME::BOARDGAME::DICE::SIZE.width) >> 1);
		position.y += ((GAME::BOARDGAME::DICE::FRAME::SIZE.height - GAME::BOARDGAME::DICE::SIZE.height) >> 1);
		al::draw_bitmap_region(bg, region, position);

		this->m_player.on_render();

		if (boardgame::internal::attack_delay > 0)
		{
			int32_t d = wind::random::generate(3);

			ALLEGRO::BITMAP pow = std::static_pointer_cast<ALLEGRO::BITMAP_DATA>(game::game_shared_data_t::get_datafile()[GAME::DATAFILE::BOARDGAME_POW]);
			ALLEGRO::RECTANGLE<int32_t> region{ { d << GAME::BOARDGAME::TILE::SHIFT.x, 0}, GAME::BOARDGAME::TILE::SIZE };
			ALLEGRO::POINT<int32_t> position = this->m_player.get_position();

			position.x -= (region.size.width >> 1);
			position.y -= (region.size.height >> 1);

			al::draw_bitmap_region(pow, region, position);
		}

		ALLEGRO::FONT font = std::static_pointer_cast<ALLEGRO::FONT_DATA>(game::game_shared_data_t::get_datafile()[GAME::DATAFILE::FONT_EMULOGIC]);
		float height = al::get_font_line_height(font);

		wind::console::draw_font(game::game_shared_data_t::get_font(), wind::map_rgb_i(0xffffff), {bg_size.width * 0.5f + 0.0f, bg_size.height - height - 4.0f + 0.0f}, WIND::CONSOLE::FONT_ALIGNMENT_CENTRE, this->m_caption.c_str());
	}

	auto boardgame_t::get_viewport() -> ALLEGRO::SIZE<int32_t>
	{
		return GAME::BUFFER::SIZE;
	}

	auto boardgame_t::get_state() const->GAME::BOARDGAME::STATE
	{
		return this->m_state;
	}

	auto boardgame_t::set_state(GAME::BOARDGAME::STATE state) -> void
	{
		this->m_state = state;
		wind::input::acknowledge();
	}
}