module;

#define ALLEGRO_UNSTABLE

#include <allegro5/allegro5.h>
#include <allegro5/allegro_opengl.h>

module application;

import <cmath>;
import <map>;
import <vector>;
import allegro;
import allegro.physfs_addon;
import allegro.image_addon;
import allegro.color_addon;
import wind;
import wind.datafile_addon;
import qrcodegen;
import game;

auto main(int32_t argc, char** argv) -> int32_t
{
	int rv = 0;
	std::shared_ptr<application::application_t> application;

	if (!al::init())
	{
		rv = -1;
	}
	else
	{
		//int32_t flags = wind::system::display::get_new_flags() | ALLEGRO::DISPLAY_FLAG_OPENGL;
		wind::system::display::set_new_flags(ALLEGRO::DISPLAY_FLAG_FULLSCREEN_WINDOW | ALLEGRO::DISPLAY_FLAG_OPENGL);

		rv = wind::run<application::application_t>(argc, argv);
	}

	return rv;
}

namespace application
{
	application_t::application_t() : m_processes(), m_process(), m_state(GAME::STATE::INVALID)
	{
	}

	application_t::~application_t()
	{
	}

	auto application_t::get_version() const -> int32_t
	{
		return GAME::VERSION;
	}

	auto application_t::get_title() const -> const wind::string_t&
	{
		static const wind::string_t title{ GAME::TITLE };


		return title;
	}

	auto application_t::on_initialize(const wind::array_t<wind::string_t>& args) -> int32_t
	{
		this->switch_state(GAME::STATE::INITIALIZATION);

		//wind::datafile::generate_header_file("assets\\index.ini", "modules\\g_datafile.ixx", WIND::DATAFILE::OUTPUT_TYPE::MODULE);
		//return -1;

		wind::lout << "Creating 8x8 font ";
		game::game_shared_data_t::m_font = wind::console::create_font_default();
		if (!game::game_shared_data_t::m_font)
		{
			wind::lout << "failed" << wind::endl;
			return -1;
		}
		wind::lout << "pass" << wind::endl;

		al::clear_to_color(wind::map_rgb_i(0x000000));
		ALLEGRO::POINT<int32_t> pos{ 16, 16 };
		wind::console::draw_font(game::game_shared_data_t::m_font, wind::map_rgb_i(0xffffff), pos, WIND::CONSOLE::FONT_ALIGNMENT_LEFT, "Loading ...");
		al::flip_display();

		wind::lout << "Creating Double Buffer: ";
		if (!(game::game_shared_data_t::m_buffer = al::create_bitmap(GAME::BUFFER::SIZE)))
		{
			wind::lout << "failed" << wind::endl;
			return -1;
		}
		wind::lout << "pass" << wind::endl;

		wind::lout << "Loading Datafile: ";
#ifdef _DEBUG
		game::game_shared_data_t::m_datafile = wind::datafile::load("assets\\index.ini");
#else
		game::game_shared_data_t::m_datafile = wind::datafile::load(GAME::ARCHIVE_FILENAME);
#endif
		if (game::game_shared_data_t::m_datafile.is_empty())
		{
			wind::lout << "failed" << wind::endl;

			while (wind::error::count())
			{
				wind::error_t e = wind::error::peek();
				wind::error::pop();
				wind::lout << e.get_message() << "\n " << e.get_filename() << "\nLine#: " << e.get_line() << "\n";
			}

			return -1;
		}
		wind::lout << "pass" << wind::endl;

		wind::lout << "Generating Title Screen State: ";
		std::shared_ptr<game::process_t> titlescreen_process = std::make_shared<game::titlescreen_t>();
		if (!titlescreen_process)
		{
			wind::lout << "failed" << wind::endl;
			return -1;
		}
		wind::lout << "pass" << wind::endl;
		this->m_processes[GAME::PROCESS::TITLESCREEN] = titlescreen_process;

		wind::lout << "Generating Options Menu State: ";
		std::shared_ptr<game::process_t> options_menu_process = std::make_shared<game::options_menu_t>();
		if (!options_menu_process)
		{
			wind::lout << "failed" << wind::endl;
			return -1;
		}
		wind::lout << "pass" << wind::endl;
		this->m_processes[GAME::PROCESS::OPTIONS_MENU] = options_menu_process;

		wind::lout << "Generating Boardgame State: ";
		std::shared_ptr<game::process_t> boardgame_process = std::make_shared<game::boardgame_t>();
		if (!boardgame_process)
		{
			wind::lout << "failed" << wind::endl;
			return -1;
		}
		wind::lout << "pass" << wind::endl;
		this->m_processes[GAME::PROCESS::BOARDGAME] = boardgame_process;

		wind::lout << "Generating Dungeon State: ";
		std::shared_ptr<game::process_t> dungeon_process = std::make_shared<game::dungeon_t>();
		if (!dungeon_process)
		{
			wind::lout << "failed" << wind::endl;
			return -1;
		}
		wind::lout << "pass" << wind::endl;
		this->m_processes[GAME::PROCESS::DUNGEON] = dungeon_process;


		if (this->load_game_data() < 0)
		{
			return -1;
		}

		al::clear_to_color(wind::map_rgb_i(0x000000));
		al::flip_display();

		this->switch_state(GAME::STATE::TITLE_SCREEN);

		return 0;
	}

	auto application_t::on_shutdown() -> int32_t
	{
		this->switch_state(GAME::STATE::SHUTDOWN);

		for (auto process : this->m_processes)
		{
			if (process.second->on_shutdown() < 0)
			{
				return -1;
			}
		}

		this->m_process.reset();
		for (auto process : this->m_processes)
		{
			process.second.reset();
		}

		game::game_shared_data_t::reset();

		return 0;
	}

	auto application_t::on_start() -> int32_t
	{
		int32_t rv{ 0 };

		if (this->m_process)
		{
			this->on_render();
			game::faders::in();
			wind::input::acknowledge();
			wind::system::timer::reset();

			rv = this->m_process->on_start();
		}

		wind::system::timer::reset();

		return rv;
	}

	auto application_t::on_stop() -> int32_t
	{
		int32_t rv{ 0 };

		if (this->m_process)
		{
			this->m_process->on_stop();
		}

		return rv;
	}

	auto application_t::on_update() -> int32_t
	{
		static ALLEGRO::SIZE<float> dsize{ 0,0 }; 
		
		dsize = al::get_display_dimensions(wind::system::display::get());

		if (this->m_process)
		{
			const ALLEGRO::SIZE<float> vsize = this->m_process->get_viewport();
			const ALLEGRO::POINT<int32_t>& mouse = wind::input::mouse::get_position();
			game::game_shared_data_t::m_mouse.x = (float)mouse.x * (vsize.width / dsize.width);
			game::game_shared_data_t::m_mouse.y = (float)mouse.y * (vsize.height / dsize.height);

			GAME::STATE state = this->m_process->on_update();

			if (state == GAME::STATE::INVALID)
			{
				return -1;
			}

			if (this->m_state != state)
			{
				this->on_render();
				game::faders::out();
				this->switch_state(state);
				if (!this->m_process)
				{
					return -1;
				}
				this->on_render();
				game::faders::in();
				wind::input::acknowledge();
				wind::system::timer::reset();
			}
		}

		for (auto process : this->m_processes)
		{
			if (process.second != this->m_process &&
				process.second->is_persistent())
			{
				if (this->m_process->on_update() == GAME::STATE::INVALID)
				{
					return -1;
				}
			}
		}

		return 0;
	}

	auto application_t::on_render() -> int32_t
	{
		ALLEGRO::BITMAP target = al::get_target_bitmap();
		al::set_target_bitmap(game::game_shared_data_t::m_buffer);
		if (this->m_process)
		{
			this->m_process->on_render();
		}
		al::set_target_bitmap(target);

		al::draw_scaled_bitmap(game::game_shared_data_t::m_buffer,
			{ {0, 0}, this->m_process->get_viewport() },
			{ { 0, 0 }, al::get_display_dimensions(wind::system::display::get()) });

		return 0;
	}

	auto application_t::switch_state(GAME::STATE state) -> void
	{
		std::shared_ptr<game::process_t> current{ this->m_process };

		if (this->m_process)
		{
			this->m_process->on_stop();
		}

		switch (state)
		{
		case GAME::STATE::INITIALIZATION:
		{
			this->m_process.reset();
		} break;
		case GAME::STATE::TITLE_SCREEN:
		{
			this->m_process = this->m_processes[GAME::PROCESS::TITLESCREEN];
		} break;
		case GAME::STATE::OPTIONS_MENU:
		{
			this->m_process = this->m_processes[GAME::PROCESS::OPTIONS_MENU];
		} break;
		case GAME::STATE::BOARDGAME:
		{
			this->m_process = this->m_processes[GAME::PROCESS::BOARDGAME];
		} break;
		case GAME::STATE::DUNGEON:
		{
			this->m_process = this->m_processes[GAME::PROCESS::DUNGEON];
		} break;
		case GAME::STATE::GAMEOVER:
		{
			this->m_process.reset();
			return;
		} break;
		case GAME::STATE::SHUTDOWN:
		{
			this->m_process.reset();
		} break;
		}

		if (this->m_process && this->m_process != current)
		{
			if (!current)
			{
				this->m_process->set_sending_process(GAME::PROCESS::APPLICATION);
			}
			else
			{
				this->m_process->set_sending_process(current->get_process());
			}
		}

		if (this->m_process)
		{
			this->m_process->on_start();
		}

		this->m_state = state;
	}

	auto application_t::load_game_data() -> int32_t
	{
#ifdef _DEBUG
		const char* current_directory = al::get_current_directory();
		al::change_directory(GAME::ASSET_DIRECTORY);
#else
		wind::string_t filename{ GAME::ARCHIVE_FILENAME };
		wind::string_t filepath{ wind::path::make_canonical(filename) };
		wind::string_t base{};
		wind::string_t ext{};
		wind::string_t path{};
		bool archive{ false };
		bool error{ false };
		int32_t rv{ -1 };

		wind::path::split_filepath(filepath, path, base, ext);

		const PHYSFS_ArchiveInfo** i{ nullptr };
		for (i = PHYSFS_supportedArchiveTypes(); *i != nullptr; i++)
		{
			wind::string_t a = wind::string::to_upper(ext);
			wind::string_t b = (*i)->extension;

			if (a == b)
			{
				archive = true;
				break;
			}
		}

		if (archive)
		{
			const ALLEGRO::FILE_INTERFACE& file_interface = al::get_new_file_interface();

			if (PHYSFS_mount(filename.c_str(), nullptr, 1))
			{
				al::physfs_addon::set_file_interface();
#endif

				for (auto process : this->m_processes)
				{
					if (process.second->on_initialize() < 0)
					{
						return -1;
					}
				}

#ifdef _DEBUG
		al::change_directory(current_directory);
#else
				PHYSFS_unmount(filename.c_str());
			}

			al::set_new_file_interface(file_interface);
		}
#endif
		return 0;
	}
}