export module game:dungeon.map;

import <cstdint>;
import <memory>;
import <map>;
import allegro;
import wind;
import :base;
import :dungeon.base;
import :dungeon.map.cell;
import :dungeon.map.camera;
import :dungeon.trigger;

namespace game
{
	namespace dungeon
	{
		namespace map
		{
			class generator_t;
		}

		export class map_t
		{
		public:
			using element_type = map::cell_t;
			using pointer_element_type = wind::add_pointer<element_type>::type;
			using const_pointer_element_type = wind::add_pointer<const element_type>::type;
			using reference_element_type = wind::add_reference<element_type>::type;
			using const_reference_element_type = wind::add_reference<const element_type>::type;

		private:
			using shared_type = std::shared_ptr<element_type[]>;

		public:
			map_t();
			map_t(const map_t& map);
			~map_t();
			auto operator = (const map_t& map)->map_t&;
			explicit operator bool() const;
			auto clear() -> void;
			auto data() -> void*;
			auto data() const -> const void*;
			auto size() const -> const ALLEGRO::SIZE<size_t>&;
			auto at(size_t index) -> reference_element_type;
			auto at(size_t index) const->const_reference_element_type;
			auto operator [](size_t index)->reference_element_type;
			auto operator [](size_t index) const->const_reference_element_type;

			class iterator
			{
			private:
				iterator() = default;
			public:
				iterator(shared_type& data, size_t offset) : m_data(data.get()), m_offset(offset) {}
				auto operator == (iterator& it) const -> bool { return (this->m_data == it.m_data && this->m_offset == it.m_offset); }
				auto operator != (iterator& it) const -> bool { return !operator == (it); }
				auto operator ++ () -> iterator& { ++this->m_offset; return *this; }
				auto operator ++ (int32_t) -> iterator { iterator tmp = *this; ++(*this); return tmp; }
				auto operator * () -> reference_element_type { return (this->m_data[this->m_offset]); }
				auto operator -> () -> pointer_element_type { return &(this->m_data[this->m_offset]); }

			private:
				const pointer_element_type m_data;
				size_t m_offset;
			};

			class const_iterator
			{
			private:
				const_iterator() = default;
			public:
				const_iterator(const shared_type& data, size_t offset) : m_data(data.get()), m_offset(offset) {}
				auto operator == (const const_iterator& it) const -> bool { return (this->m_data == it.m_data && this->m_offset == it.m_offset); }
				auto operator != (const const_iterator& it) const -> bool { return !operator == (it); }
				auto operator ++ () -> const_iterator& { ++this->m_offset; return *this; }
				auto operator ++ (int32_t) -> const_iterator { const_iterator tmp = *this; ++(*this); return tmp; }
				auto operator * () const -> const_reference_element_type { return (this->m_data[this->m_offset]); }
				auto operator -> () const -> const_pointer_element_type { return &(this->m_data[this->m_offset]); }

			private:
				const pointer_element_type m_data;
				size_t m_offset;
			};

			auto begin() -> iterator;
			auto end() -> iterator;
			auto cbegin() const->map_t::const_iterator;
			auto cend() const->map_t::const_iterator;
			auto get_start() const -> const ALLEGRO::POINT<int32_t>&;
			auto get_finish() const -> const ALLEGRO::POINT<int32_t>&;
			auto get_boss_door() const -> const ALLEGRO::POINT<int32_t>&;
			auto get_index(const ALLEGRO::POINT<int32_t>& point) const->int32_t;
			auto draw(const std::shared_ptr<wind::tilesheet_t>& tilesheet, const map::camera_t& camera, const ALLEGRO::POINT<int32_t>& position) const -> void;
			auto unlock_room(const ALLEGRO::POINT<int32_t>& point) -> void;
			auto generate_new(bool is_bottom_floor = false) -> int32_t;

			auto get_cell(const ALLEGRO::POINT<int32_t>& point, GAME::DIRECTION direction) -> map::cell_t&;
			auto get_cell(const ALLEGRO::POINT<int32_t>& point, GAME::DIRECTION direction) const-> const map::cell_t&;

			auto get_trigger_map() const -> const trigger_map_t&;
			auto get_trigger_map() -> trigger_map_t&;

			friend class map::generator_t;

		private:
			auto reset(size_t width, size_t height) -> bool;
			auto reset(const ALLEGRO::SIZE<size_t>& size) -> bool;

			shared_type m_data{};
			trigger_map_t m_trigger_map{};
			ALLEGRO::SIZE<size_t> m_size{ 0, 0 };
			ALLEGRO::POINT<int32_t> m_start{ 0, 0 };
			ALLEGRO::POINT<int32_t> m_finish{ 0, 0 };
			ALLEGRO::POINT<int32_t> m_boss{ 0, 0 };
		};
	}
}