unit Player;
(*<Implements player stuff. *)
(* Copyright (c) 2024 Guillermo Martínez J.

  This software is provided 'as-is', without any express or implied
  warranty. In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you must not
    claim that you wrote the original software. If you use this software
    in a product, an acknowledgment in the product documentation would be
    appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and must not be
    misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
    distribution.
 *)
interface

  uses
    Classes,
    CastleControls, CastleTransform, CastleUIControls,
    CharacterDefinition, NavigationBehavior;

  type
  (* The player behavior.

     Assign to the @link(TCastleTransform) that will represent the player in the
     view.
   *)
    TPlayerBehavior = class (TNavigationBehavior)
    private
      fTimeCooldown: Single;
    public
    (* Constructor. *)
      constructor Create (aOwner: TComponent); override;
    (* Does an attack. *)
      procedure Attack;
    (* Check if player is attacking. *)
      function CanDoThings: Boolean; override;
    (* Player was hurted. *)
      procedure Hurt;
    (* Updates the behavior. *)
      procedure Update (
        const aSecondsPassed: Single;
        var aRemoveMe: TRemoveType
      ); override;
    end;



  (* Manages player data. *)
    TPlayerData = class (TObject)
    private
      fOwnsBehavior: Boolean;
      fStats: TCharacterValues;
    (* Reference to objects in current view. *)
      fBehavior: TPlayerBehavior;
      fImgBlood, fImgAttackFail, fImgAttackHit: TCastleImageControl;

      fLabelHealth: TCastleLabel;
    public
    (* Constructor. *)
      constructor Create;
    (* Destructor. *)
      destructor Destroy; override;

    (* Create player at game beginning. *)
      procedure InitGame;
    (* Init the player avatar and connects with elements. *)
      procedure InitView (
        aAvatar: TCastleTransform;
        aControlPanel, aEffectImages: TCastleUserInterface
      );
    (* Get attacked by enemies or weapons or whatever. *)
      procedure Attacked (const aAttacker: TCharacterValues);
    (* Interacts whith something. *)
      procedure Interact;

    (* To be used with events. *)
      procedure ClickMoveLeft (aSender: TObject);
      procedure ClickMoveBackward (aSender: TObject);
      procedure ClickMoveRight (aSender: TObject);
      procedure ClickMoveForward (aSender: TObject);
      procedure ClickRotateLeft (aSender: TObject);
      procedure ClickRotateRight (aSender: TObject);
      procedure ClickTurnAround (aSender: TObject);

      procedure ClickInteract (aSender: TObject);
      procedure ClickAttack (aSender: TObject);

    (* Reference to the behavior.

       This shouldn exist but I'm in a hurry.
     *)
      property Behavior: TPlayerBehavior read fBehavior;
    (* Player stats. *)
      property Stats: TCharacterValues read fStats;
    (* Reference to effect images. *)
      property ImgBlood: TCastleImageControl read fImgBlood write fImgBlood;
      property ImgAttackHit: TCastleImageControl
        read fImgAttackHit write fImgAttackHit;
      property ImgAttackFail: TCastleImageControl
        read fImgAttackFail write fImgAttackFail;
    end;

  var
  (* The game data.

    It is global because it should exist in all views.

    Use it as read-only.
   *)
    PlayerData: TPlayerData;

implementation

  uses
    sysutils,
    CastleLog, CastleComponentSerialize, CastleUtils,
    GameViewDied, GameViewSnowtown,
    EnemyBehavior, ItemBehavior;

  const
    PlayerSpeed = 0.25;
    PlayerStats: TCharacterValues = (
      Health:   100;
      Strength:  10;
      Defense:    0;
      Armour:     0
    );

(*
 * TPlayerBehavior
 *************************************************************************)

  constructor TPlayerBehavior.Create (aOwner: TComponent);
  begin
    inherited Create (aOwner);
    Self.Speed := PlayerSpeed;
    fTimeCooldown := 0
  end;



  procedure TPlayerBehavior.Attack;
  var
    lItem: TCastleTransform;
    lEnemy: TEnemyBehavior;
  begin
    if not Self.CanDoThings then Exit;
    lItem := Self.LookingNearby;
  { Did hit? }
    if Assigned (lItem) then
    begin
      PlayerData.ImgAttackHit.Exists := True;
      lEnemy := TEnemyBehavior (lItem.FindBehavior (TEnemyBehavior));
      if Assigned (lEnemy) then
        lEnemy.Attacked (PlayerData.Stats)
    end
    else
      PlayerData.ImgAttackFail.Exists := True;
  { Cooldown, in any case. }
    fTimeCooldown := Self.Speed
  end;



  function TPlayerBehavior.CanDoThings: Boolean;
  begin
    Result := not (Self.IsMoving or (fTimeCooldown > 0))
  end;



  procedure TPlayerBehavior.Hurt;
  begin
    PlayerData.ImgBlood.Exists := True;
    fTimeCooldown := Self.Speed
  end;



  procedure TPlayerBehavior.Update (
    const aSecondsPassed: Single;
    var aRemoveMe: TRemoveType
  );
  begin
    if CastleApplicationMode <> appRunning then Exit;
    inherited Update (aSecondsPassed, aRemoveMe);
  { Attack cooldown. }
    if Self.CanDoThings then
    begin
      PlayerData.ImgBlood.Exists := False;
      PlayerData.ImgAttackFail.Exists := False;
      PlayerData.ImgAttackHit.Exists := False
    end
    else
      fTimeCooldown := fTimeCooldown - aSecondsPassed
  end;



(*
 * TPlayerData
 *************************************************************************)

  constructor TPlayerData.Create;
  begin
    inherited Create;
    fOwnsBehavior := False
  end;



  destructor TPlayerData.Destroy;
  begin
    if fOwnsBehavior then fBehavior.Free;
    inherited Destroy;
  end;



  procedure TPlayerData.InitGame;
  begin
    fStats := PlayerStats
  end;



  procedure TPlayerData.InitView (
    aAvatar: TCastleTransform;
    aControlPanel, aEffectImages: TCastleUserInterface
  );

    procedure AttackEffects;
    var
      lNdx: Integer;
    begin
      for lNdx := aEffectImages.ControlsCount - 1 downto 0 do
      begin
        if aEffectImages.Controls[lNdx].Name = 'imgBlood' then
          fImgBlood := TCastleImageControl (aEffectImages.Controls[lNdx]);
        if aEffectImages.Controls[lNdx].Name = 'imgAttackFail' then
          fImgAttackFail := TCastleImageControl (aEffectImages.Controls[lNdx]);
        if aEffectImages.Controls[lNdx].Name = 'imgAttackHit' then
          fImgAttackHit := TCastleImageControl (aEffectImages.Controls[lNdx])
      end;
      try
        fImgBlood.Exists := False;
        fImgAttackFail.Exists := False;
        fImgAttackHit.Exists := False
      except
        on Error: Exception do
        begin
          WritelnLog ('AttackEffects', 'Revise attack effect images!!!');
          raise Error
        end
      end;
    end;

    procedure ControlPanel;

      procedure SetButtonClick (
        aControl: TCastleUserInterface;
        aMethod: TNotifyEvent
      ); inline;
      var
        lButton: TCastleButton absolute aControl;
      begin
        lButton.OnClick := aMethod;
      end;

      procedure MovementPanel (aPanel: TCastleUserInterface);
      var
        lNdx: Integer;
      begin
        for lNdx := aPanel.ControlsCount - 1 downto 0 do
        begin
          if aPanel.Controls[lNdx].Name = 'btnMoveLeft' then
            SetButtonClick (aPanel.Controls[lNdx], {$ifdef FPC}@{$endif} Self.ClickMoveLeft);
          if aPanel.Controls[lNdx].Name = 'btnMoveForward' then
            SetButtonClick (aPanel.Controls[lNdx], {$ifdef FPC}@{$endif} Self.ClickMoveForward);
          if aPanel.Controls[lNdx].Name = 'btnMoveRight' then
            SetButtonClick (aPanel.Controls[lNdx], {$ifdef FPC}@{$endif} Self.ClickMoveRight);
          if aPanel.Controls[lNdx].Name = 'btnMoveBackward' then
            SetButtonClick (aPanel.Controls[lNdx], {$ifdef FPC}@{$endif} Self.ClickMoveBackward);

          if aPanel.Controls[lNdx].Name = 'btnRotateLeft' then
            SetButtonClick (aPanel.Controls[lNdx], {$ifdef FPC}@{$endif} Self.ClickRotateLeft);
          if aPanel.Controls[lNdx].Name = 'btnRotateRight' then
            SetButtonClick (aPanel.Controls[lNdx], {$ifdef FPC}@{$endif} Self.ClickRotateRight);
          if aPanel.Controls[lNdx].Name = 'btnTurnAround' then
            SetButtonClick (aPanel.Controls[lNdx], {$ifdef FPC}@{$endif} Self.ClickTurnAround)
        end
      end;

    var
      lNdx: Integer;
    begin
      for lNdx := aControlPanel.ControlsCount - 1 downto 0 do
      begin
        if aControlPanel.Controls[lNdx].Name = 'MovementPanel' then
          MovementPanel (aControlPanel.Controls[lNdx]);
        if aControlPanel.Controls[lNdx].Name = 'btnInteract' then
          SetButtonClick (aControlPanel.Controls[lNdx], {$ifdef FPC}@{$endif} Self.ClickInteract);
        if aControlPanel.Controls[lNdx].Name = 'btnAttack' then
          SetButtonClick (aControlPanel.Controls[lNdx], {$ifdef FPC}@{$endif} Self.ClickAttack);
        if aControlPanel.Controls[lNdx].Name = 'lblHealth' then
          fLabelHealth := TCastleLabel (aControlPanel.Controls[lNdx])
      end
    end;

  begin
  { Check if it has behavior and assign as needed. }
    if Assigned (aAvatar.FindBehavior (TPlayerBehavior)) then
    begin
    { If owns behavior, free it as we don't need it anymore. }
      if fOwnsBehavior and Assigned (fBehavior) then fBehavior.Free;
      fBehavior := TPlayerBehavior (aAvatar.FindBehavior (TPlayerBehavior));
      fOwnsBehavior := False
    end
    else
    begin
    { We need a behavior. }
      if not fOwnsBehavior then
      begin
        fBehavior := TPlayerBehavior.Create (Nil);
        fOwnsBehavior := True
      end;
      aAvatar.AddBehavior (fBehavior)
    end;
    ControlPanel;
    AttackEffects
  end;



  procedure TPlayerData.Attacked (const aAttacker: TCharacterValues);
  var
    lDamage: Integer;
  begin
    lDamage := DoAttack (aAttacker, fStats);
    if lDamage > 0 then
    begin
      fBehavior.SnapPosition;
      Dec (fStats.Health, lDamage);
      if Assigned (fLabelHealth) then
        fLabelHealth.Caption := Format ('Health: %d', [fStats.Health]);
      if fStats.Health > 0 then fBehavior.Hurt
    end;
  end;



  procedure TPlayerData.Interact;
  var
    lItem: TCastleTransform;
  begin
    lItem := fBehavior.LookingNearby;
    if Assigned (lItem) and (LeftStr (lItem.Name, 6) = 'Health')
    and (fStats.Health < 100)
    then
    begin
      lItem.Exists := False;
      Inc (fStats.Health, 25);
      if Assigned (fLabelHealth) then
        fLabelHealth.Caption := Format ('Health: %d', [fStats.Health])
    end
  end;



  procedure TPlayerData.ClickMoveLeft (aSender: TObject);
  begin
    fBehavior.Move (IncreaseDirection (fBehavior.Direction, ToLeft))
  end;



  procedure TPlayerData.ClickMoveBackward (aSender: TObject);
  begin
    fBehavior.Move (IncreaseDirection (fBehavior.Direction, 2))
  end;



  procedure TPlayerData.ClickMoveRight (aSender: TObject);
  begin
    fBehavior.Move (IncreaseDirection (fBehavior.Direction, ToRight))
  end;



  procedure TPlayerData.ClickMoveForward (aSender: TObject);
  begin
    fBehavior.Move (fBehavior.Direction)
  end;



  procedure TPlayerData.ClickRotateLeft (aSender: TObject);
  begin
    fBehavior.Rotate (ToLeft)
  end;



  procedure TPlayerData.ClickRotateRight (aSender: TObject);
  begin
    fBehavior.Rotate (ToRight)
  end;



  procedure TPlayerData.ClickTurnAround (aSender: TObject);
  begin
    fBehavior.TurnAround
  end;



  procedure TPlayerData.ClickInteract (aSender: TObject);
  begin
    Self.Interact
  end;



  procedure TPlayerData.ClickAttack (aSender: TObject);
  begin
    fBehavior.Attack
  end;

initialization
  PlayerData := TPlayerData.Create;
{ Register behavior to be available from CGE editor. }
  RegisterSerializableComponent (TPlayerBehavior, 'Player')
finalization
  PlayerData.Free
end.
