UNIT mngFSM;
(*<Simple implementation of the state pattern.

   This is a generic implementation of the Finite State Machine.  A lot of it
   was taken from "Programming Game AI by Example" book by Mat Buckland.

   To use this FSM, expand @link(TmngState) class and implement the
   @link(TmngState.Execute) method.  Then the object that wants to use the FSM
   should create a @link(TmngFiniteStateMachine) object assigning the apropriate
   owner.  Then assign the state and call the @link(TmngFiniteStateMachine.Run)
   method.

   This is a little different than @link(mngGameState), as this is designed for
   game entities while @code(mngGameState) is designed for game states (title,
   configuration, play...).

   NOTE:  This unit was modified to fit in the CGE.
 *)
(*
  Copyright (c) 2010-2020 Guillermo Martínez J.

  This software is provided 'as-is', without any express or implied
  warranty. In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you must not
    claim that you wrote the original software. If you use this software
    in a product, an acknowledgment in the product documentation would be
    appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and must not be
    misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
    distribution.
 *)

{$I mngconfig.inc }

INTERFACE

  USES
    sysutils;

  TYPE
  (* Exception raised by state machine. *)
    mngStateMachineException = CLASS (Exception);



  (* Base class for states.
    @seealso(TmngFiniteStateMachine) *)
    TmngState = CLASS (TObject)
    PUBLIC
    (* Executed when entering in this state.  By default it does nothing.
       @param(aEntity Object asociated with the state.)
       @seealso(Leave) *)
      PROCEDURE Enter (aEntity: TObject); VIRTUAL;
    (* Implements the state logic.

       Descendent classes must implement this method.
       @param(aEntity Object asociated with the state.)
     *)
      PROCEDURE Execute (aEntity: TObject);
        VIRTUAL; ABSTRACT;
    (* Executed when exiting from this state.  By default it does nothing.
       @param(aEntity Object asociated with the state.)
       @seealso(Enter) *)
      PROCEDURE Leave (aEntity: TObject); VIRTUAL;
    END;



  (* The Finite State Machine itself.

     Note that it doesn't own any state.  This means it doesn't destroy the
     state objects when it's destroyed.
     @seealso(TmngState) *)
    TmngFiniteStateMachine = CLASS (TObject)
    PRIVATE
      fOwner: TObject;
      fCurrentState, fPreviousState, fGlobalState: TmngState;

      PROCEDURE SetState (aState: TmngState);
      PROCEDURE SetGlobal (aState: TmngState);
    PUBLIC
    (* Creates the state machine.
      @param(aOwner The owner of the state machine.  It should not be @nil.)
      @param(aGlobal A global state that will be executed every time.  It may
             be @nil)
     *)
      CONSTRUCTOR Create (aOwner: TObject);
    (* Executes current state and global state if any. *)
      PROCEDURE Run; INLINE;
    (* Sets current state to the previous state.  It allows to swap between
       two states. @seealso(Current) *)
      PROCEDURE SetPreviousState; INLINE;

    (* Reference to the object that owns this state machine. *)
      PROPERTY Owner: TObject READ fOwner;
    (* Current state. @seealso(SetPreviousState) *)
      PROPERTY Current: TmngState READ fCurrentState WRITE SetState;
    (* A global state that will be executed every time.  It may be @nil. *)
      PROPERTY Global: TmngState READ fGlobalState WRITE SetGlobal;
    END;

IMPLEMENTATION

  const
    SET_NIL_STATE = 'Trying to set NIL state!';
    STATE_OWNER_NOT_NULL = 'State machine owner shouldn''t be NIL.';

(*
 * TmngState
 *****************************************************************************)

(* Executed when the FSM enters in this state. *)
  PROCEDURE TmngState.Enter (aEntity: TObject); BEGIN aEntity := aEntity END;

(* Executed just when the FSM exits from this state. *)
  PROCEDURE TmngState.Leave (aEntity: TObject); BEGIN aEntity := aEntity END;



(*
 * TmngFiniteStateMachine
 *****************************************************************************)

(* Changes current state. *)
  PROCEDURE TmngFiniteStateMachine.SetState (aState: TmngState);
  BEGIN
    IF aState = NIL THEN
    { It would be "assign a DoNothingState", but may be there's a badly
      created/destroyed object, so an exception should hint that something is
      wrong somewhere. }
      RAISE mngStateMachineException.Create (SET_NIL_STATE);
    IF fCurrentState <> NIL THEN fCurrentState.Leave (SELF.fOwner);
    fPreviousState := fCurrentState;
    fCurrentState := aState;
    fCurrentState.Enter (SELF.fOwner)
  END;



(* Changes global state. *)
  PROCEDURE TmngFiniteStateMachine.SetGlobal (aState: TmngState);
  BEGIN
    IF fGlobalState <> NIL THEN fGlobalState.Leave (SELF.fOwner);
    fGlobalState := aState;
    IF fGlobalState <> NIL THEN fGlobalState.Enter (SELF.fOwner)
  END;



(* Creates the FSM. *)
  CONSTRUCTOR TmngFiniteStateMachine.Create (aOwner: TObject);
  BEGIN
    INHERITED Create;
    IF aOwner = NIL THEN
      RAISE mngStateMachineException.Create (STATE_OWNER_NOT_NULL);
    fOwner := aOwner;
    fGlobalState := NIL; fCurrentState := NIL
  END;



(* Runs state. *)
  PROCEDURE TmngFiniteStateMachine.Run;
  BEGIN
    IF fGlobalState <> NIL THEN
      fGlobalState.Execute (SELF.fOwner);
    IF fCurrentState <> NIL THEN
      fCurrentState.Execute (SELF.fOwner)
  END;



(* Sets current state to the previous state.  It allows to swap between
   two states. *)
  PROCEDURE TmngFiniteStateMachine.SetPreviousState;
  BEGIN
    SetState (fPreviousState)
  END;

END.
