unit GameViewSnowtown;
(***<A snowy town. *)
(* Copyright (c) 2024 Guillermo Martínez J.

  This software is provided 'as-is', without any express or implied
  warranty. In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you must not
    claim that you wrote the original software. If you use this software
    in a product, an acknowledgment in the product documentation would be
    appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and must not be
    misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
    distribution.
 *)
interface

  uses
    Classes,
    CastleComponentSerialize, CastleUIControls, CastleControls,
    CastleKeysMouse, CastleViewport, CastleScene, CastleVectors, CastleCameras,
    CastleTransform,
    EnemyBehavior;

type
  { Main "playing game" view, where most of the game logic takes place. }
  TViewSnowtown = class (TCastleView)
  published
    { Components designed using CGE editor.
      These fields will be automatically initialized at Start. }
    Player: TCastleTransform;
    AttackEffects, ControlPanel: TCastleRectangleControl;

    LabelFps: TCastleLabel;
    MainViewport: TCastleViewport;
    MainFog: TCastleFog;
    btnMoveLeft, btnMoveBackward, btnMoveRight, btnMoveForward: TCastleButton;
    btnRotateLeft, btnRotateRight, btnTurnAround: TCastleButton;
    lblDirection: TCastleLabel;
    btnInteract, btnAttack: TCastleButton;
    ListEnemies, ListItems: TCastleTransform;
  private
    procedure RotationChange (aSender: TObject);
    procedure StartCooldown (aSender: TObject);
    procedure EndCooldown (aSender: TObject);
  public
    constructor Create(AOwner: TComponent); override;
    procedure Start; override;
    procedure Stop; override;
    procedure Update(const SecondsPassed: Single; var HandleInput: Boolean); override;
    function Press(const Event: TInputPressRelease): Boolean; override;
  end;

var
  ViewSnowtown: TViewSnowtown;

implementation

uses
  SysUtils, Math,
ItemBehavior,
  CastleLog, CastleStringUtils, CastleFilesUtils, CastleUtils,
  GameViewMenu, GameViewDied, Orcs, Player;

{ TViewSnowtown ----------------------------------------------------------------- }

  procedure TViewSnowtown.RotationChange (aSender: TObject);
  begin
    lblDirection.Caption := 'Looking at: ' + PlayerData.Behavior.CurrentDirectionName;
  end;



  procedure TViewSnowtown.StartCooldown (aSender: TObject);
  begin
WritelnLog ('StartCooldown');
    Self.btnMoveLeft.Enabled := False;
    Self.btnMoveBackward.Enabled := False;
    Self.btnMoveRight.Enabled := False;
    Self.btnMoveForward.Enabled := False;
    Self.btnRotateLeft.Enabled := False;
    Self.btnRotateRight.Enabled := False;
    Self.btnTurnAround.Enabled := False;
    Self.btnInteract.Enabled := False;
    Self.btnAttack.Enabled := False
  end;



  procedure TViewSnowtown.EndCooldown (aSender: TObject);
  begin
WritelnLog ('EndCooldown');
    Self.btnMoveLeft.Enabled := True;
    Self.btnMoveBackward.Enabled := True;
    Self.btnMoveRight.Enabled := True;
    Self.btnMoveForward.Enabled := True;
    Self.btnRotateLeft.Enabled := True;
    Self.btnRotateRight.Enabled := True;
    Self.btnTurnAround.Enabled := True;
    Self.btnInteract.Enabled := True;
    Self.btnAttack.Enabled := True
  end;



  constructor TViewSnowtown.Create(AOwner: TComponent);
  begin
    inherited;
    DesignUrl := 'castle-data:/gameviewsnowtown.castle-user-interface'
  end;



  procedure TViewSnowtown.Start;

    procedure InitPlayer;
    var
      lBehavior: TPlayerBehavior;
    begin
      PlayerData.InitView (Self.Player, ControlPanel, AttackEffects);
      PlayerData.Behavior.OnChangeDirection :=  {$IFDEF FPC}@{$ENDIF}Self.RotationChange;
      lblDirection.Caption := 'Looking at: ' + PlayerData.Behavior.CurrentDirectionName;
    { This only first time. }
      WritelnLog ('ViewPlay::Start', 'PlayerData.InitGame');
      PlayerData.InitGame
    end;

    procedure InitItems;
    var
      lItem: TCastleTransform;
    begin
      for lItem in Self.ListItems do
        lItem.AddBehavior (TItemBehavior.Create (Self.FreeAtStop))
    end;

  begin
    inherited Start;
    InitPlayer;
    InitItems;
    Orcs.Initialize (Self.ListEnemies, Self.FreeAtStop);
    Randomize
  end;



  procedure TViewSnowtown.Stop;
  begin
    inherited Stop;
  end;



  procedure TViewSnowtown.Update (const SecondsPassed: Single; var HandleInput: Boolean);

    procedure MovementButtonsEnabled; inline;
    var
      lValue: Boolean;
    begin
      lValue := PlayerData.Behavior.CanDoThings;
      Self.btnMoveLeft.Enabled := lValue;
      Self.btnMoveBackward.Enabled := lValue;
      Self.btnMoveRight.Enabled := lValue;
      Self.btnMoveForward.Enabled := lValue;
      Self.btnRotateLeft.Enabled := lValue;
      Self.btnRotateRight.Enabled := lValue;
      Self.btnTurnAround.Enabled := lValue
    end;

    procedure InteractionButtonsEnabled; inline;
    var
      lValue: Boolean;
    begin
      lValue := PlayerData.Behavior.CanDoThings;
      Self.btnInteract.Enabled := lValue;
      Self.btnAttack.Enabled := lValue
    end;

  begin
    inherited Update (SecondsPassed, HandleInput);
  { This virtual method is executed every frame (many times per second). }
    Assert (LabelFps <> nil, 'If you remove LabelFps from the design, remember to remove also the assignment "LabelFps.Caption := ..." from code');
    LabelFps.Caption := 'FPS: ' + Container.Fps.ToString;
  { Not very happy whit this, but I'm in a hurry. }
    if PlayerData.Stats.Health < 1 then
    begin
      Self.Container.View := ViewDied;
      Exit
    end;
    MovementButtonsEnabled;
    InteractionButtonsEnabled
  end;



  function TViewSnowtown.Press (const Event: TInputPressRelease): Boolean;
  begin
  { Allow the ancestor to handle keys. }
    if inherited Press (Event) then Exit;
  { Function keys. }
    if Event.IsKey (keyF5) then
    begin
      Container.SaveScreenToDefaultFile;
      Exit (True)
    end;
    if Event.IsKey (keyEscape) then
    begin
      Container.View := ViewMenu;
      Exit (True)
    end;
  { Movement keys. }
    if PlayerData.Behavior.IsMoving then Exit;
    if Event.IsKey(keyW) then
    begin
      PlayerData.ClickMoveForward (Self);
      Exit (True)
    end;
    if Event.IsKey(keyS) then
    begin
      PlayerData.ClickMoveBackward (Self);
      Exit (True)
    end;
    if Event.IsKey(keyD) then
    begin
      PlayerData.ClickMoveRight (Self);
      Exit (True)
    end;
    if Event.IsKey(keyA) then
    begin
      PlayerData.ClickMoveLeft (Self);
      Exit (True)
    end;

    if Event.IsKey(keyQ) then
    begin
      PlayerData.ClickRotateLeft (Self);
      Exit (True)
    end;
    if Event.IsKey(keyE) then
    begin
      PlayerData.ClickRotateRight (Self);
      Exit (True)
    end
  end;

end.
