unit NavigationBehavior;
(*<Implements a @italic(behavior) that allows to navigate. *)
(* Copyright (c) 2024 Guillermo Martínez J.

  This software is provided 'as-is', without any express or implied
  warranty. In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you must not
    claim that you wrote the original software. If you use this software
    in a product, an acknowledgment in the product documentation would be
    appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and must not be
    misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
    distribution.
 *)

interface

  uses
    Classes,
    CastleTransform, CastleClassUtils, CastleVectors;

  const
  (* Current direction. *)
    ToFront = 0;
  (* Current right direction. *)
    ToRight = 1;
  (* Current left direction. *)
    ToLeft = -1;
  (* Size of a grid tile. *)
    GridSize = 1;

  type
  (* Direction expressed a 4 possible horizontal directions.  They're in
       clockwise order.
    @seealso(DirectionName) @seealso(DirectionVector)
   *)
    TNavDirection = (dirNorth, dirEast, dirSouth, dirWest);



  (* Allows navigation.
   *)
    TNavigationBehavior = class (TCastleBehavior)
    private
    (* Current state. *)
      fIsMoving: Boolean;
      fDirection: TNavDirection;
    (* Animation information. *)
      fEndDirection: TNavDirection;
      fSpeed, fCurrentTime: Single;
      fBeginPosition, fEndPosition: TVector3;
      fTurningAround: Boolean;
      fTurningDirection: Integer;

      fOnChangeDirection, fOnChangeMoving, fOnShoved: TNotifyEvent;
    protected
    (* Assigns @link(Direction). *)
      procedure SetDirection (const aDirection: TNavDirection); virtual;

    (* Adjust parent position in the grid-map, and also position. *)
      procedure FixParentPosition; virtual;
    published
    (* Direction it's looking.  Synced with @code(Parent.Direction).
      @seealso(Rotate)
     *)
      property Direction: TNavDirection
        read fDirection write SetDirection
        default dirNorth;
    (* How much time needed to do a movement.
      @seealso(Move) @seealso(Rotate)
     *)
      property Speed: Single read fSpeed write fSpeed default 1;

    (* Event called when direction changes. *)
      property OnChangeDirection: TNotifyEvent
        read fOnChangeDirection write fOnChangeDirection;
    (* Event called when starts or stops moving. *)
      property OnChangeMoving: TNotifyEvent
        read fOnChangeMoving write fOnChangeMoving;
    (* Event called when entity is pushed back.  This may happend when it
       crosses the path of another entity.
     *)
      property OnShoved: TNotifyEvent read fOnShoved write fOnShoved;
    public
    (* Constructor. *)
      constructor Create (aOwner: TComponent); override;

    (* Check if entity can do an action. *)
      function CanDoThings: Boolean; virtual;
    (* Check if there's an obstacle in front of the entity. *)
      function CannotWalk (aPos1, apos2: TVector3): Boolean; overload;
      function CannotWalk (const aDirection: Integer = ToFront): Boolean; overload;
      function CannotWalkTo (const aDest: TVector3): Boolean;
    (* Returns the object that is in front or @nil if there's none. *)
      function LookingNearby (const aDirection: Integer = ToFront)
        : TCastleTransform;
    (* Called when assigning the @code(Parent). *)
      procedure ParentAfterAttach; override;
    (* Define property sections in the editor. *)
      function PropertySections (const aPropertyName: String): TPropertySections;
        override;
    (* Snap position to nearest grid. *)
      procedure SnapPosition;
    (*
      Updates the behavior.  Called every game @italic(tick).
      @param(aSecondsPassed Time passed since previous call.)
      @param(aRemoveMe Asign @code(rtRemove) or @code(rtRemoveAndFree) to remove
             this component from @code(Parent).)
     *)
    procedure Update (
      const aSecondsPassed: Single;
      var aRemoveMe: TRemoveType
    ); override;
  (* Tells the behavior that it should rotate in the given direction.

     It doesn't rotate if it is moving.
     @seealso(Update)
     @seealso(Speed) @seealso(IsMoving) @seealso(Direction)
     @seealso(OnChangeMoving)
     @seealso(GridSize)
   *)
    procedure Rotate (const aRotationChange: Integer); virtual;
  (* Does a 180 degrees turn. *)
    procedure TurnAround;
  (* Tells the behavior that it should move one grid tile in the given
     direction.

     It doesn't move if it is moving.
     @seealso(Update)
     @seealso(Speed) @seealso(IsMoving)
     @seealso(OnChangeMoving)
     @seealso(GridSize)
   *)
    procedure Move (const aMoveDirection: TNavDirection); virtual;
  (* Tells the behavior that it should move one grid tile down. *)
    procedure MoveDown;
  (* Return current direction name. *)
    function CurrentDirectionName: String; inline;

  (* Returns @true if it is moving the parent.
    @seealso(Move) @seealso(Rotate)  @seealso(fOnChangeMoving)
   *)
    property IsMoving: Boolean read fIsMoving;
  end;


  const
  (* Name of directions. @seealso(TNavDirection) *)
    DirectionName: array [TNavDirection] of String = (
      'North',
      'East',
      'South',
      'West'
    );
  (* Vectors pointing in each direction. @seealso(TNavDirection) *)
    DirectionVector: array [TNavDirection] of TVector3 = (
      (X:  0; Y: 0; Z: -1),
      (X:  1; Y: 0; Z:  0),
      (X:  0; Y: 0; Z:  1),
      (X: -1; Y: 0; Z:  0)
    );

(* Helper to make rotations.
  @param(aDirection The direction it's pointing.)
  @param(aIncrement The way it's rotating.)
  @seealso(ToRight) @seealso(Toleft)
 *)
  function IncreaseDirection (
    const aDirection: TNavDirection;
    const aIncrement: Integer
  ): TNavDirection; inline;

implementation

  uses
    CastleComponentSerialize, CastleLog, CastleUtils;

  function IncreaseDirection (
    const aDirection: TNavDirection;
    const aIncrement: Integer
  ): TNavDirection;
  inline;
  begin
    Result := TNavDirection (ChangeIntCycle (
      Ord (aDirection),
      aIncrement,
      Ord (High (TNavDirection))
    ))
  end;



(*
 * TNavigationBehavior
 *************************************************************************)

  procedure TNavigationBehavior.SetDirection (const aDirection: TNavDirection);
  begin
    if fDirection = aDirection then Exit;
    fDirection := aDirection; fEndDirection := aDirection;
    if Assigned (Self.Parent) then
      Self.Parent.Direction := DirectionVector[aDirection];
    if Assigned (fOnChangeDirection) then
      fOnChangeDirection (Self)
  end;



  procedure TNavigationBehavior.FixParentPosition;
  var
    lPosition, lDirection, lUp: TVector3;
  begin
    if Assigned (Self.Parent) then
    begin
      Self.Parent.GetView (lPosition, lDirection, lUp);
      lDirection := DirectionVector[fDirection];
      lPosition.X := Trunc (lPosition.X + 0.5) - 0.5;
      lPosition.Z := Trunc (lPosition.Z + 0.5) - 0.5;
      Self.Parent.SetView (lPosition, lDirection, lUp)
    end
  end;



  constructor TNavigationBehavior.Create (aOwner: TComponent);
  begin
    inherited Create (aOwner);
    fTurningAround := False;
    fIsMoving := False;
  { Default values. }
    fDirection := dirNorth;
    fSpeed := 1
  end;



  function TNavigationBehavior.CanDoThings: Boolean;
  begin
    Result := not fIsMoving
  end;



  function TNavigationBehavior.CannotWalk (aPos1, apos2: TVector3): Boolean;
  begin
  { Don't stop by small objects (some objects have the origin in the bottom). }
    aPos1.Y := 0.4;
    aPos2.Y := 0.4;
  { Don't collide with yourself. }
    Self.Parent.Exists := False;
    Result := Self.World.WorldSegmentCollision (aPos1, aPos2);
  { You exists. }
    Self.Parent.Exists := True
  end;



  function TNavigationBehavior.CannotWalk (const aDirection: Integer): Boolean;
  var
    lDirection: TNavDirection;
  begin
    lDirection := IncreaseDirection (fDirection, aDirection);
    Result := Self.CannotWalk (
      fBeginPosition,
      fBeginPosition + DirectionVector[lDirection]
    )
  end;



  function TNavigationBehavior.CannotWalkTo (const aDest: TVector3): Boolean;
  begin
    Result := Self.CannotWalk (fBeginPosition, aDest)
  end;



  function TNavigationBehavior.LookingNearby (const aDirection: Integer)
    : TCastleTransform;
  var
    lRayOrigin, lRayDirection: TVector3;
    lDistance: Single;
  begin
    lRayDirection := DirectionVector[IncreaseDirection (fDirection, aDirection)];
  { Ray a little avobe the ground to "see" the small objects (i.e. presents) and
    in front of current position to avoid detecting items in the current
    position.
  }
    lRayOrigin := Self.Parent.Translation + (lRayDirection * (GridSize * 0.5));
    lRayOrigin.Y := 0.1;
  { Find objects. }
    Self.Parent.Exists := False;
    try
      try
        Result := Self.World.WorldRayCast (
          lRayOrigin,
          lRayDirection,
          lDistance
        )
      finally
        Self.Parent.Exists := True
      end
    except
      WritelnLog ('error', 'CanInteract: Worldraycast failed');
      Result := Nil
    end;
  { Only objects nearby. }
    if Assigned (Result) and (lDistance > GridSize) then Result := Nil
  end;



  procedure TNavigationBehavior.ParentAfterAttach;
  begin
    inherited ParentAfterAttach;
    Self.FixParentPosition
  end;



  function TNavigationBehavior.PropertySections (const aPropertyName: String)
    : TPropertySections;
  const
    BasicProperties: array [0..2] of String = (
      'Name', 'Speed', 'Tag'
    );
  begin
    if ArrayContainsString (aPropertyName, BasicProperties) then
      Result := [psBasic]
    else
    if aPropertyName = 'Direction' then
      Result := [psBasic, psLayout]
    else
      Result := inherited PropertySections (aPropertyName)
  end;



  procedure TNavigationBehavior.SnapPosition;
  var
    lPosition, lDirection, lUp: TVector3;
  begin
    if Assigned (Self.Parent) then
    begin
      Self.Parent.GetView (lPosition, lDirection, lUp);
      lDirection := DirectionVector[fDirection];
      lPosition.X := Trunc (lPosition.X + 0.5) - 0.5;
      lPosition.Z := Trunc (lPosition.Z + 0.5) - 0.5;
      Self.Parent.SetView (lPosition, lDirection, lUp)
    end
  end;



  procedure TNavigationBehavior.Update (
    const aSecondsPassed: Single;
    var aRemoveMe: TRemoveType
  );
  var
    lPosition, lDirection, lUp: TVector3;
    lProportion: Double;

    procedure DoRotation; inline;

      procedure DoTurnAround; inline;
      var
        lTmp: Single;
      begin
      { Double speed, because it must do two rotations in one. }
        lTmp := lProportion;
        lProportion := lProportion * 2;
        if lProportion > 1 then
        begin
        { It's the second half. }
          lProportion := lProportion - 1;
          if fTurningDirection <> 0 then
          begin
          { It didn't started the second half yet. }
            lTmp := fCurrentTime;
            fDirection := fEndDirection;
            fIsMoving := False;
            Self.Rotate (fTurningDirection);
            fTurningAround := True;
            fTurningDirection := 0;
            fCurrentTime := lTmp
          end
        end
      end;

    begin
      if fTurningAround then DoTurnAround;
      Self.Parent.SetView (
        lPosition,
        lDirection.Lerp (
          lProportion,
          DirectionVector[fDirection],
          DirectionVector[fEndDirection]
        ),
        lUp
      )
    end;

    procedure DoTranslation; inline;
    begin
    { To avoid to cross other entity path. }
      if Self.CannotWalkTo (fEndPosition) then
      begin
      { Back to initial position. }
        Self.Parent.SetView (fBeginPosition, DirectionVector[fDirection], lUp);
        fEndPosition := fBeginPosition;
        fIsMoving := False;
        if Assigned (fOnShoved) then fOnShoved (Self);
        if Assigned (fOnChangeMoving) then fOnChangeMoving (Self);
        Exit
      end;
      Self.Parent.SetView (
        fBeginPosition.Lerp (lProportion, fBeginPosition, fEndPosition),
        lDirection,
        lUp
      )
    end;

  begin
    if not (fIsMoving and (CastleApplicationMode = appRunning)) then
      Exit;
  { Get current state. }
    Self.Parent.GetView (lPosition, lDirection, lUp);
  { Time passes...}
    fCurrentTime := fCurrentTime + aSecondsPassed;
    if fCurrentTime < fSpeed then
    begin
    { Lets move. }
      lProportion := fCurrentTime / fSpeed;
      if fDirection = fEndDirection then DoTranslation else DoRotation
    end
    else
    begin
    { Last step. }
      Self.SetDirection (fEndDirection);
      Self.Parent.SetView (fEndPosition, DirectionVector[fDirection], lUp);
      fIsMoving := False;
      if Assigned (fOnChangeMoving) then fOnChangeMoving (Self)
    end
  end;



  procedure TNavigationBehavior.Rotate (const aRotationChange: Integer);
  begin
    if Self.CanDoThings then
    begin
    { Get begin position and assigns end position to prevent translation. }
      fBeginPosition := Self.Parent.Translation;
      fEndPosition := fBeginPosition;
    { Where to rotate. }
      fEndDirection := IncreaseDirection (fDirection, aRotationChange);
      fCurrentTime := 0;
    { Movement is ready. }
      fIsMoving := True; fTurningAround := False;
      if Assigned (fOnChangeMoving) then fOnChangeMoving (Self)
    end
  end;



  procedure TNavigationBehavior.TurnAround;
  begin
    if Self.CanDoThings then
    begin
      if Random (2) = 1 then
        fTurningDirection := ToRight
      else
        fTurningDirection := ToLeft;
      Self.Rotate (fTurningDirection);
      fTurningAround := True
    end
  end;



  procedure TNavigationBehavior.Move (const aMoveDirection: TNavDirection);
  begin
    if Self.CanDoThings then
    begin
    { Current position. }
      fBeginPosition := Self.Parent.Translation;
    { Where to move. }
      fEndPosition := fBeginPosition + DirectionVector[aMoveDirection] * GridSize;
    { Check if there is an obstacle. }
      if Self.CannotWalkTo (fEndPosition) then Exit;
      fCurrentTime := 0;
    { Movement is ready. }
      fIsMoving := True;
      if Assigned (fOnChangeMoving) then fOnChangeMoving (Self)
    end
  end;



  procedure TNavigationBehavior.MoveDown;
  const
    GoingDown: TVector3 = (X:  0; Y: -1; Z:  0);
  begin
    fBeginPosition := Self.Parent.Translation;
    fEndPosition := fBeginPosition + GoingDown * GridSize;
    fCurrentTime := 0;
    fIsMoving := True
  end;



  function TNavigationBehavior.CurrentDirectionName: String;
  begin
    Result := DirectionName[fDirection]
  end;

initialization
{Optional:  To make this behavior available from the editor add
 NavigationBehavior to editor_units="..." in CastleEngineManifest.xml and reopen
 the project.
}
  RegisterSerializableComponent (TNavigationBehavior, 'Navigation');
end.
