unit EnemyBehavior;
(*<Implements the basic enemy @italic(behavior). *)
(* Copyright (c) 2024 Guillermo Martínez J.

  This software is provided 'as-is', without any express or implied
  warranty. In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you must not
    claim that you wrote the original software. If you use this software
    in a product, an acknowledgment in the product documentation would be
    appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and must not be
    misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
    distribution.
 *)

interface

  uses
     CharacterBehavior, CharacterDefinition;

  type
  (* The enemy behavior.

     Default behavior is wander the map until it's near the player.  Then it
     tries to find him and attack when it's in front of him..
   *)
    TEnemyBehavior = class (TCharacterBehavior)
    private
      fData: TCharacterValues;
      fActionDirection: Integer;
      fHurtState, fDieState: TCharacterSate;
    public
    (* Enemy is attacked... *)
      procedure Attacked (const aAttacker: TCharacterValues);

    (* Enemy stats. *)
      property Stats: TCharacterValues read fData write fData; { TODO: Read-only? }
    (* Action direction.  Used by some states. *)
      property ActionDirection: Integer
        read fActionDirection write fActionDirection;
    (* State used when it's attacked. *)
      property HurtState: TCharacterSate read fHurtState write fHurtState;
    (* State used when die. *)
      property DieState: TCharacterSate read fDieState write fDieState;
    end;



  (* Enemy waits until he/she finds the player or is tired of wait. *)
    TEnemyWaitState = class (TWaitState)
    private
      fWanderState: TCharacterSate;
    protected
    (* Called when wants to think. *)
      procedure Think (aBehavior: TCharacterBehavior); override;
    public
    (* State used to walk. *)
      property WanderState: TCharacterSate read fWanderState write fWanderState;
    end;



  (* The enemy attacks the player if he/she is nearby.  Else, it changes to
     the @link(EndAttackState) state.

     Note this state uses the @code(Animation) property for the attack action,
     using 'idle' while "thinking".
   *)
    TEnemyAttack = class (TCharacterSate)
    private
      fStopState: TCharacterSate;
    public
    { Initialize behavior. }
      procedure Enter (aEntity: TObject); override;
    (* Implements the state logic. *)
      procedure Execute (aEntity: TObject); override;

    (* State to stop the attack. *)
      property StopState: TCharacterSate read fStopState write fStopState;
    end;



  (* The enemy is looking around for the player. *)
    TEnemyWatchState = class (TCharacterSate)
    private
      fAttackState: TCharacterSate;
    public
    (* Implements the state logic. *)
      procedure Execute (aEntity: TObject); override;

    (* State to change when it finds the player. *)
      property AttackState: TCharacterSate read fAttackState write fAttackState;
    end;

implementation

  uses
     sysutils,
CastleLog,
    CastleComponentSerialize, CastleScene, CastleTransform,
    NavigationBehavior, Player;

(*
 * TEnemyBehavior
 *************************************************************************)

  procedure TEnemyBehavior.Attacked (const aAttacker: TCharacterValues);
  var
    lDamage: Integer;
  begin
    lDamage := DoAttack (aAttacker, fData);
    if lDamage > 0 then
    begin
      Self.SnapPosition;
      Dec (fData.Health, lDamage);
      if fData.Health <= 0 then
        Self.StateMachine.Current := fDieState
      else
        Self.StateMachine.Current := fHurtState
    end
  end;




(*
 * TEnemyWaitState
 *************************************************************************)

  procedure TEnemyWaitState.Think (aBehavior: TCharacterBehavior);
  begin
    if Random > 0.1 then aBehavior.StateMachine.Current := fWanderState
  end;



(*
 * TEnemyAttack
 *************************************************************************)

  procedure TEnemyAttack.Enter(aEntity: TObject);
  var
    lCharacterBehavior: TCharacterBehavior absolute aEntity;
  begin
    TCastleScene (lCharacterBehavior.Parent).PlayAnimation ('idle', True)
  end;



  procedure TEnemyAttack.Execute (aEntity: TObject);
  var
    lCharacterBehavior: TEnemyBehavior absolute aEntity;

    procedure DoAttack; inline;
    begin
    { NOTE:  this doesn't mean you can use "with" anywhere!!! }
      with TCastleScene (lCharacterBehavior.Parent) do
      begin
        PlayAnimation ('idle', true);
        PlayAnimation (Self.Animation,false);
      end;
      PlayerData.Attacked (lCharacterBehavior.Stats)
    end;

    function ObjectiveInRange: Boolean; inline;
    var
      lObject: TCastleTransform;
    begin
      lObject := lCharacterBehavior.LookingNearby;
      Result := Assigned (lObject) and (lObject.Name = '')
    end;

    function PointToObjective: Boolean; inline;
    begin
      Result := lCharacterBehavior.ActionDirection = 0
    end;

    procedure RotateToObjective; inline;
    begin
      TCastleScene (lCharacterBehavior.Parent).PlayAnimation ('walk', true);
      lCharacterBehavior.Rotate (lCharacterBehavior.ActionDirection);
      lCharacterBehavior.ActionDirection := 0;
    end;

  begin
    if PointToObjective then
    begin
      if ObjectiveInRange then
        DoAttack
      else
        lCharacterBehavior.StateMachine.Current := fStopState
    end
    else
      RotateToObjective
  end;



(*
 * TEnemyWatchState
 *************************************************************************)

  procedure TEnemyWatchState.Execute (aEntity: TObject);
  var
    lCharacterBehavior: TEnemyBehavior absolute aEntity;
    lObject: TCastleTransform;
    lDirection: Integer;
  begin
    if lCharacterBehavior.StateMachine.Current = fAttackState then Exit;
    for lDirection := ToLeft to ToRight do
    begin
      lObject := lCharacterBehavior.LookingNearby (lDirection);
    { WARNING:

      To know if it "sees" the player it should use:

      if Assigned (lObject) and (lObject.FindBehavior (TPlayerBehavior) then

      but as you see, it assumes that when object doesn't have a name then it's
      the player.  It shouldn't be this way but for some reason LookingNearby
      returns an unnamed object when it "sees" the Player object.  That object
      isn't the Player as it doesn't have the TPlayerBehavior nor any child.

      Should ask with CGE developers about this.
    }
      if Assigned (lObject) and (lObject.Name = '')
      then
      begin
        lCharacterBehavior.ActionDirection := lDirection;
        lCharacterBehavior.StateMachine.Current := fAttackState;
        Exit
      end
    end
  end;

initialization
{ Register behavior to be available from CGE editor. }
  RegisterSerializableComponent (TEnemyBehavior, 'Enemy')
finalization
  ;
end.
