unit CharacterBehavior;
(*<Implements the basic character @italic(behavior). *)
(* Copyright (c) 2024 Guillermo Martínez J.

  This software is provided 'as-is', without any express or implied
  warranty. In no event will the authors be held liable for any damages
  arising from the use of this software.

  Permission is granted to anyone to use this software for any purpose,
  including commercial applications, and to alter it and redistribute it
  freely, subject to the following restrictions:

    1. The origin of this software must not be misrepresented; you must not
    claim that you wrote the original software. If you use this software
    in a product, an acknowledgment in the product documentation would be
    appreciated but is not required.

    2. Altered source versions must be plainly marked as such, and must not be
    misrepresented as being the original software.

    3. This notice may not be removed or altered from any source
    distribution.
 *)

interface

  uses
    Classes,
    mngFSM, CastleClassUtils, CastleTransform,
    NavigationBehavior;

  const
  (* Default @link(TCharacterBehavior.TurnTime). *)
    DefaultTurnTime = 1;

  type
  (* Base class for character states. *)
    TCharacterSate = class (TmngState)
    private
      fAnimation: String;
      fAnimationLoop: Boolean;
      fLoopAnimation: Boolean;
    public
    (* Constructor. *)
      constructor Create; virtual;
    (* Executed when entering in this state.  It sets the @link(Animation)
       animation.
     *)
      procedure Enter (aEntity: TObject); override;
    (* Implements the state logic.  Does nothing. *)
      procedure Execute (aEntity: TObject); override;

    (* Animation to use. *)
      property Animation: String read fAnimation write fAnimation;
    (* Does animation loops.  Default is @true. *)
      property AnimationLoop: Boolean
        read fAnimationLoop write fAnimationLoop;
    end;



  (* The character behavior.

     Default behavior is wander the map.
   *)
    TCharacterBehavior = class (TNavigationBehavior)
    private
      fStateMachine: TmngFiniteStateMachine;
      fTurnTime, fTimeTilTurn: Single;
      fTag: Integer;
    published
    (* Time between turns, in seconds. *)
      property TurnTime: Single read fTurnTime write fTurnTime default DefaultTurnTime;
    public
    (* Constructor. *)
      constructor Create (aOwner: TComponent); override;
    (* Destructor. *)
      destructor Destroy; override;
    (* Called when assigning the @code(Parent).

       By default it sets the state @link(StateWander) *)
      procedure ParentAfterAttach; override;
    (*
      Updates the behavior.  Called every game @italic(tick).
      @param(aSecondsPassed Time passed since previous call.)
      @param(aRemoveMe Asign @code(rtRemove) or @code(rtRemoveAndFree) to remove
      this component from @code(Parent).)
     *)
    (* Define property sections in the editor. *)
      function PropertySections (const aPropertyName: String): TPropertySections;
        override;
    procedure Update (
      const aSecondsPassed: Single;
      var aRemoveMe: TRemoveType
    ); override;

  (* Access to the state machine. *)
    property StateMachine: TmngFiniteStateMachine read fStateMachine;
  (* A value used by some states. *)
    property Tag: Integer read fTag write fTag;
  end;



  (* The character waits until something happens.  Uses animation 'idle'. *)
    TWaitState = class (TCharacterSate)
    private
      fMaxTurns: Integer;
    protected
    (* Called when wants to think.  By default does nothing. *)
      procedure Think (aBehavior: TCharacterBehavior); virtual;
    public
    (* Constructor. *)
      constructor Create; override;
    (* Setup state. *)
      procedure Enter (aEntity: TObject); override;
    (* Implements the state logic. *)
      procedure Execute (aEntity: TObject); override;

    (* How many turns wait to call @link(Think). *)
      property NumTurns: Integer read fMaxTurns write fMaxTurns;
    end;



  (* A state that makes the character to wander around the map.
   *)
    TWalkState = class (TCharacterSate)
    private
      fMaxSteps: Integer;
      fStateStop: TCharacterSate;

    (* Change animation. *)
      procedure MovingStateChanged (aSender: TObject);
    protected
    (* Used when entity is blocked and cannot walk to the front.

       By default, it rotates in a random direction or turn-around. *)
      procedure ExitBlock (aEntity: TCharacterBehavior); virtual;
    public
    (* Constructor. *)
      constructor Create; override;
    (* Setup state. *)
      procedure Enter (aEntity: TObject); override;
    (* Implements the state logic. *)
      procedure Execute (aEntity: TObject); override;
    (* Exit state. *)
      procedure Leave (aEntity: TObject); override;

    (* Max number of steps will walk/rotate.  Default is 1. *)
      property MaxSteps: Integer read fMaxSteps write fMaxSteps;
    (* State to change when it stops walking. *)
      property StopState: TCharacterSate read fStateStop write fStateStop;
    end;



  (* A state that plays the animation for a while and changes to another state.
   *)
    TInterludeState = class (TCharacterSate)
    private
      fContinueState: TmngState;
      fTurnsWaiting: Integer;
    public
    (* Init state. *)
      procedure Enter (aEntity: TObject); override;
    (* Updates state. *)
      procedure Execute (aEntity: TObject); override;

    (* Number of turns waiting.  Default is 1. *)
      property Turns: Integer read fTurnsWaiting write fTurnsWaiting;
    (* The state to change. *)
      property ContinueState: TmngState read fContinueState write fContinueState;
    end;



  (* A state used when character dies.

     It plays the @italic('die') animation once, then waits and disapears.
   *)
    TDieState = class (TCharacterSate)
    private
      fTimeVanish: Integer;
    public
    (* Constructor. *)
      constructor Create; override;
    (* Init state. *)
      procedure Enter (aEntity: TObject); override;
    (* Updates state. *)
      procedure Execute (aEntity: TObject); override;

    (* Number of turns it lies until it vanishes.  Default is 2. *)
      property TimeVanish: Integer read fTimeVanish write fTimeVanish;
    end;

implementation

  uses
CastleLog,
    CastleComponentSerialize, CastleScene, CastleUtils, CastleVectors;

(*
 * TCharacterSate
 *************************************************************************)

  constructor TCharacterSate.Create;
  begin
    inherited Create;
    fAnimation := 'idle';
    fAnimationLoop := True
  end;



  procedure TCharacterSate.Enter (aEntity: TObject);
  var
    lCharacterBehavior: TCharacterBehavior absolute aEntity;
  begin
    if fAnimation <> '' then
      TCastleScene (lCharacterBehavior.Parent).PlayAnimation (
        fAnimation,
        fLoopAnimation
      )
  end;



  procedure TCharacterSate.Execute (aEntity: TObject);
  begin
    ; { Does nothing. }
  end;



(*
 * TCharacterBehavior
 *************************************************************************)

  constructor TCharacterBehavior.Create (aOwner: TComponent);
  begin
    inherited Create (aOwner);
    fStateMachine := TmngFiniteStateMachine.Create (Self);
  { Default values. }
    fTurnTime := DefaultTurnTime;
    fTag := 0
  end;



  destructor TCharacterBehavior.Destroy;
  begin
    fStateMachine.Free;
    inherited Destroy
  end;



  procedure TCharacterBehavior.ParentAfterAttach;
  begin
    inherited ParentAfterAttach;
    fTimeTilTurn := fTurnTime;
  { Next should make it a bit more efficent. }
    TCastleScene (Self.Parent).AnimateOnlyWhenVisible := True
  end;



  function TCharacterBehavior.PropertySections (const aPropertyName: String
    ): TPropertySections;
  begin
    if aPropertyName = 'TurnTime' then
      Result := [psBasic]
    else
      Result := inherited PropertySections (aPropertyName)
  end;



  procedure TCharacterBehavior.Update (
    const aSecondsPassed: Single;
    var aRemoveMe: TRemoveType
  );
  begin
    if CastleApplicationMode <> appRunning then Exit;
    inherited Update (aSecondsPassed, aRemoveMe);
  { Execute AI only in turn. }
    fTimeTilTurn := fTimeTilTurn - aSecondsPassed;
    if (fTimeTilTurn > 0) or Self.IsMoving then Exit;
    fStateMachine.Run;
    fTimeTilTurn := fTurnTime
  end;


(*
 * TWaitState
 *************************************************************************)

  procedure TWaitState.Think (aBehavior: TCharacterBehavior);
  begin
    WritelnLog ('TStateWaiting', 'Thinking...')
  end;



  constructor TWaitState.Create;
  begin
    inherited Create;
    fMaxTurns := 1
  end;



  procedure TWaitState.Enter (aEntity: TObject);
  var
     lCharacterBehavior: TCharacterBehavior absolute aEntity;
  begin
    inherited Enter (aEntity);
    lCharacterBehavior.Tag := 0
  end;



  procedure TWaitState.Execute (aEntity: TObject);
  var
    lCharacterBehavior: TCharacterBehavior absolute aEntity;
  begin
    lCharacterBehavior.Tag := lCharacterBehavior.Tag + 1;
    if lCharacterBehavior.Tag = fMaxTurns then
    begin
      lCharacterBehavior.Tag := 0;
      Self.Think (lCharacterBehavior)
    end
  end;



(*
 * TWalkState
 *************************************************************************)

  procedure TWalkState.MovingStateChanged (aSender: TObject);
  var
    lCharacterBehavior: TCharacterBehavior absolute aSender;
  begin
    if lCharacterBehavior.IsMoving then
      TCastleScene (lCharacterBehavior.Parent).PlayAnimation ('walk', True)
    else
      TCastleScene (lCharacterBehavior.Parent).PlayAnimation ('idle', True)
  end;



  procedure TWalkState.ExitBlock (aEntity: TCharacterBehavior);
  const
    MaxIterations = 5;
  var
    lRepetions, lRotation: Integer;
  begin
    lRepetions := 1; lRotation := 0;
    while (lRepetions < MaxIterations) and aEntity.CannotWalk (lRotation) do
    begin
      if Random < 0.5 then lRotation := ToRight else lRotation := ToLeft;
      Inc (lRepetions)
    end;
    if (lRepetions < MaxIterations) and (lRotation <> 0) then
      aEntity.Rotate (lRotation)
    else
      aEntity.TurnAround
  end;



  constructor TWalkState.Create;
  begin
    inherited Create;
    fMaxSteps := 1
  { Do not set Animation := 'walk' or it will sart walking in the site one turn.
  }
  end;



  procedure TWalkState.Enter (aEntity: TObject);
  var
    lCharacterBehavior: TCharacterBehavior absolute aEntity;
  begin
    inherited Enter (aEntity);
    lCharacterBehavior.OnChangeMoving :=
      {$IFDEF FPC}@{$ENDIF}Self.MovingStateChanged;
  { How many steps to walk. }
    lCharacterBehavior.Tag := RandomIntRange (1, fMaxSteps)
  end;



  procedure TWalkState.Execute (aEntity: TObject);
  var
    lCharacterBehavior: TCharacterBehavior absolute aEntity;

    function ChangeDirection: Boolean;
    var
      lRotation: Integer;
    begin
      if Random > 0.5 then lRotation := ToRight else lRotation := ToLeft;
      if lCharacterBehavior.CannotWalk (lRotation) then Exit (False);
      lCharacterBehavior.Rotate (lRotation);
      Result := True
    end;

  begin
    if lCharacterBehavior.IsMoving then Exit;
    if lCharacterBehavior.Tag > 0 then
    begin
      lCharacterBehavior.Tag := lCharacterBehavior.Tag - 1;
    { Check if there's something blocking the way. }
      if lCharacterBehavior.CannotWalk then
        Self.ExitBlock (lCharacterBehavior)
      else
    { Change direction once each four. }
      if RandomIntRange (1, 4) = 2 then
        if ChangeDirection then
          Exit;
    { In any other case, walk. }
      lCharacterBehavior.Move (lCharacterBehavior.Direction)
    end
    else
      lCharacterBehavior.StateMachine.Current := fStateStop
  end;



  procedure TWalkState.Leave (aEntity: TObject);
  var
    lCharacterBehavior: TCharacterBehavior absolute aEntity;
  begin
    lCharacterBehavior.OnChangeMoving := Nil
  end;



(*
 * TInterludeState
 ************************************************************************)

  procedure TInterludeState.Enter(aEntity: TObject);
  var
    lBehavior: TCharacterBehavior absolute aEntity;
  begin
    inherited Enter (aEntity);
    fTurnsWaiting := 1;
    lBehavior.Tag := 0
  end;



  procedure TInterludeState.Execute ( aEntity: TObject);
  var
    lBehavior: TCharacterBehavior absolute aEntity;
  begin
    lBehavior.Tag := lBehavior.Tag + 1;
    if lBehavior.Tag >= fTurnsWaiting then
      lBehavior.StateMachine.Current := fContinueState
  end;



(*
 * TDieState
 ************************************************************************)
  constructor TDieState.Create;
  begin
    inherited Create;
    Self.Animation := 'die';
    Self.AnimationLoop := False;
    Self.TimeVanish := 2
  end;



  procedure TDieState.Enter (aEntity: TObject);
  var
    lBehavior: TCharacterBehavior absolute aEntity;
  begin
    lBehavior.StateMachine.Global := Nil;
    inherited Enter (aEntity);
    lBehavior.Tag := fTimeVanish
  end;



  procedure TDieState.Execute (aEntity: TObject);
  var
    lBehavior: TCharacterBehavior absolute aEntity;
  begin
    lBehavior.Tag := lBehavior.Tag - 1;
    if lBehavior.Tag = 0 then lBehavior.MoveDown;
    if lBehavior.Tag < 0 then lBehavior.Parent.Exists := False
  end;

initialization
{ Register behavior to be available from CGE editor. }
  RegisterSerializableComponent (TCharacterBehavior, 'Character');
finalization
  ;
end.
