#include "game.h"

bool Game::exit;

void Game::MainLoop()
{
    double previousTick = al_get_time();
    double accumulatedTicks = 0.0;
    const double LOGIC_TICK_PERIOD = Timer::LOGIC_TICK_PERIOD;
    const double ELASPED_TICKS_MAX = 0.25;

#define DEBUG_FPS_COUNTER /// Performance testing. Comment out to disable.
#ifdef DEBUG_FPS_COUNTER
    double testFpsTicks = 0;
    double testFrameCount = 0;
    double testFpsResult = 0;
#endif

    while(!exit)
    {
        double currentTick = al_get_time();
        double elaspedTicks = currentTick - previousTick;
        previousTick = currentTick;

        while (!al_is_event_queue_empty(Event::eventQueue))
        {
            al_get_next_event(Event::eventQueue, &Event::event);
            switch(Event::event.type)
            {
            case ALLEGRO_EVENT_DISPLAY_CLOSE:
                exit = true;
                break;
            case ALLEGRO_EVENT_KEY_DOWN:
                Keyboard::InputKeydown();
                break;
            case ALLEGRO_EVENT_KEY_UP:
                Keyboard::InputKeyup();
                break;
            case ALLEGRO_EVENT_MOUSE_AXES:
                Mouse::InputMouseXY();
                Mouse::InputMousewheel();
                break;
            }
        }

        if(elaspedTicks > ELASPED_TICKS_MAX)
            elaspedTicks = ELASPED_TICKS_MAX; // Avoid long frame skips?

        accumulatedTicks += elaspedTicks;

        while(accumulatedTicks >= LOGIC_TICK_PERIOD)
        {
            InputSwitchboard();
            LogicSwitchboard();
            Audio::Logic();
            Mouse::mouseAxesAltered = false;
            accumulatedTicks -= LOGIC_TICK_PERIOD;
        }

        Timer::UpdateInterpolationFactor(accumulatedTicks/LOGIC_TICK_PERIOD);

        static double lastRenderTick = 0.0;
        if(currentTick - lastRenderTick >= Timer::maxFrameDuration)
        {
            DrawingSwitchboard(); // Hopefully limited by vsync.
            lastRenderTick = currentTick;
#ifdef DEBUG_FPS_COUNTER
            testFrameCount ++;
#endif
        }

#ifdef DEBUG_FPS_COUNTER
        if(currentTick - testFpsTicks >= 1.0)
        {
            testFpsResult = testFrameCount;
            testFrameCount = 0;
            testFpsTicks = currentTick;
            std::cout << "t = " << al_get_time() << ", FPS: " << testFpsResult << std::endl;
        }
#endif
    }
}

bool Game::Initialize(char **argv)
{
    exit = false;

    srand(time(NULL));

    al_install_system(ALLEGRO_VERSION_INT, NULL);
    Display::Initialize();
    Keyboard::Initialize();
    Mouse::Initialize();
    Timer::Initialize();
    Event::Initialize();

    Filesystem::Initialize(argv);
    Filesystem::SetPhysfsFileInterface();
    Image::Initialize();
    Shader::Initialize();
    Font::Initialize();
    Audio::Initialize();

    Filesystem::SetStandardFileInterface();

    Palette::InitializeEulbink();
    Scene::Initialize();
    Camera::Initialize();

    // *Might* not want to initialize everything, later.
    Title::Initialize();
    Overworld::Initialize();
    Settings::Initialize();
    //Archive::Initialize();

    return true;
}

void Game::Uninitialize()
{
    //Archive::Uninitialize();
    Overworld::Uninitialize();

    Camera::Uninitialize();

    Audio::Uninitialize();
    Font::Uninitialize();
    Image::Uninitialize();
    Shader::Uninitialize();
    Filesystem::Uninitialize();

    Event::Uninitialize();
    Timer::Uninitialize();
    Mouse::Uninitialize();
    Keyboard::Uninitialize();
    Display::Uninitialize();

    al_uninstall_system();

}


void Game::InputSwitchboard()
{
    Keyboard::InputKeyHold();
    Mouse::InputMouseButtonHold();

    switch(Scene::inputContext)
    {
    case Scene::INPUT_CONTEXT_TITLE:
        Title::Input();
        break;
    case Scene::INPUT_CONTEXT_OVERWORLD:
        Overworld::Input();
        break;
        /*
    case Scene::INPUT_CONTEXT_ARCHIVE:
        Archive::Input();
        break;
        */
    case Scene::INPUT_CONTEXT_SETTINGS:
        Settings::Input();
        break;
    }
}

void Game::LogicSwitchboard()
{
    switch(Scene::scene)
    {
    case Scene::SCENE_TITLE:
        Title::Logic();
        if(Title::exitOptionSelected)
            exit = true;
        break;
    case Scene::SCENE_OVERWORLD:
        Overworld::Logic();
        break;
    case Scene::SCENE_SETTINGS:
        Settings::Logic();
        break;
        /*
    case Scene::SCENE_ARCHIVE:
        Archive::Logic();
        break;
        */
    }
}

void Game::DrawingSwitchboard()
{
    al_clear_to_color(COLKEY_BACKGROUND);

    switch(Scene::scene)
    {
    case Scene::SCENE_TITLE:
        Title::Drawing();
        break;
    case Scene::SCENE_OVERWORLD:
        Overworld::Drawing();
        break;
    case Scene::SCENE_SETTINGS:
        Settings::Drawing();
        break;
        /*
    case Scene::SCENE_ARCHIVE:
        Archive::Drawing();
        break;
        */
    }

    al_flip_display();
}
