#include "emitter.h"

Emitter::Emitter()
{

}

Emitter::Emitter(int emitter_preset)
{
    Initialize(emitter_preset);
}

Emitter::~Emitter()
{
    //std::cout << "emitter destruction test";
}

void Emitter::Initialize(int emitter_preset)
{
    Actor::Initialize();

    SetIsOnline(true);
    SetIsNPCEmitter(false);
    SetTrackedTarget(nullptr);

    emitterPreset = emitter_preset;
    bulletPreset = EmitterIndex::emitterBulletPreset.at(emitterPreset);

    //std::cout << "bullet preset id: " << emitterPreset << std::endl;

    willLockTrackedPosition = EmitterIndex::emitterWillLockTrackedPosition.at(emitterPreset);
    isTrackedPositionLocked = false;
    //SetTrackedXYPosition(0,0);

    isSpawnFiringDelay = true;
    spawnFiringDelay = DEFAULT_SPAWN_FIRING_DELAY;

    isReloading = false;

    volleySize = EmitterIndex::emitterVolleySize.at(emitterPreset);
    if(volleySize == 0)
        volleySize = 1;

    volleyArcWidth = EmitterIndex::emitterArcWidth.at(emitterPreset);

    currentVolley = 0;
    numVolleys = std::get<EmitterIndex::LABEL_NUM_VOLLEYS>( EmitterIndex::emitterVolleyRepeatData.at(emitterPreset) );
    interVolleyTicks = 0;
    interVolleyTicksRequired = std::get<EmitterIndex::LABEL_INTER_VOLLEY_TICKS_REQUIRED>( EmitterIndex::emitterVolleyRepeatData.at(emitterPreset) );
    reloadTicks = 0;
    reloadTicksRequired = std::get<EmitterIndex::LABEL_RELOAD_TICKS_REQUIRED>( EmitterIndex::emitterVolleyRepeatData.at(emitterPreset) );

    isTelegraphActive = EmitterIndex::emitterTelegraph.at(emitterPreset).first;
    isTelegraphVisible = false;
    telegraphColour = EmitterIndex::emitterTelegraph.at(emitterPreset).second;
    telegraphScaling = 0.0;

    if(telegraphColour == EmitterIndex::TELEGRAPH_COLOUR_BLACK)
        telegraphScalingFactor = 0.48;
    else
        telegraphScalingFactor = 0.32;

    telegraphRotationOffset = 0.0;

    SetNumFrames(EmitterIndex::NUM_EMITTER_TELEGRAPH_FRAMES);
}

void Emitter::Logic()
{
    Actor::Logic();

    if(hasTrackedPosition)
    {
        float opposite = trackedYPosition - GetYPosition();
        float adjacent = trackedXPosition - GetXPosition();

        volleyAngle = std::atan2( opposite, adjacent );
    }
    else if(hasTrackedTarget)
    {
        float opposite = GetTrackedTarget()->GetYPosition() - GetYPosition();
        float adjacent = GetTrackedTarget()->GetXPosition() - GetXPosition();

        volleyAngle = std::atan2( opposite, adjacent );
    }

    if(! isSpawnFiringDelay)
    {
        if(currentVolley < numVolleys)
        {
            if(interVolleyTicks < interVolleyTicksRequired)
                interVolleyTicks ++;
            if(interVolleyTicks >= interVolleyTicksRequired)
            {
                if(isOnline)
                {
                    Fire();

                    interVolleyTicks = 0;
                    currentVolley ++;

                    if(currentVolley == numVolleys)
                    {
                        reloadTicks = 0;
                        isReloading = true;
                        if(willLockTrackedPosition) // Break tracking until next set of shots.
                        {
                            hasTrackedPosition = false;
                            isTrackedPositionLocked = false;
                        }

                    }
                }
            }
        }
        else //currentVolley >= numVolleys
        {
            if(reloadTicks < reloadTicksRequired)
            {
                reloadTicks ++;
                if(reloadTicks >= reloadTicksRequired)
                {
                    currentVolley = 0;
                    reloadTicks = 0;
                    isReloading = false;
                }
            }
        }
    }
    else // isSpawnFiringDelay
    {
        spawnFiringDelay --;
        if(spawnFiringDelay <= 0)
            isSpawnFiringDelay = false;
    }

    if(isTelegraphActive)
    {
        if(isSpawnFiringDelay && spawnFiringDelay < TELEGRAPH_TICKS_MAX)
        {
            telegraphScaling = 1.0 + spawnFiringDelay*telegraphScalingFactor;
            telegraphRotationOffset = spawnFiringDelay * 2*ALLEGRO_PI/Timer::LPS;
            isTelegraphVisible = true;
        }
        else if(isReloading && (reloadTicksRequired-reloadTicks) < TELEGRAPH_TICKS_MAX)
        {
            telegraphScaling = 1.0 + (reloadTicksRequired-reloadTicks )*telegraphScalingFactor;
            telegraphRotationOffset = (reloadTicksRequired-reloadTicks) * 2*ALLEGRO_PI/Timer::LPS;
            isTelegraphVisible = true;
        }
        else
            isTelegraphVisible = false;

    }
}

void Emitter::Fire()
{
    //std::cout << "BOOM TIME" << std::endl;
    //Audio::AddSfx(Audio::genericLaserShootWav);


    if(willLockTrackedPosition && !isTrackedPositionLocked) // The first Fire() will set the tracked position for all following Fire() until reload.
    {
        if(hasTrackedTarget && GetTrackedTarget() != nullptr)
        {
            SetHasTrackedPosition(true);
            SetTrackedXYPosition(GetTrackedTarget()->GetXPosition(), GetTrackedTarget()->GetYPosition() ); // This only works for ships, not arbitrary positions.
            isTrackedPositionLocked = true;
        }
        else
        {
            // do nothing.
        }
    }


    float patternStartAngle = volleyAngle - volleyArcWidth/2;
    float seperationAngle = volleyArcWidth / volleySize;

    for(unsigned bulletPosition = 0; bulletPosition < volleySize; bulletPosition++)
    {
        Bullet*bullet;
        bullet = new Bullet();

        bullet->Initialize(bulletPreset, patternStartAngle + seperationAngle * (bulletPosition + 0.5));
        //std::cout << "Test bullet preset: " << bulletPreset << std::endl;
        bullet->SetAllXYPositions(GetXPosition(), GetYPosition());

        if(isNPCEmitter)
            bullet->SetIsNPCBullet(true);

        Bullet::bullets.push_back(bullet);

        if(bullet->GetIsPersistent())
        {
            Audio::AddSfx(Audio::firePersistent0);
        }
        else
            Audio::AddSfx(Audio::fireTransient0);
    }
}

void Emitter::DrawTelegraph(float ship_move_angle)
{
    UpdateInterpolation();

    if(isTelegraphActive)
    {
        if(isTelegraphVisible)
        {
            al_draw_scaled_rotated_bitmap(Image::emitterTelegraphSub[telegraphColour*EmitterIndex::NUM_EMITTER_TELEGRAPH_FRAMES + GetCurrentFrame()],
                                          GetSpriteWidth()/2, GetSpriteHeight()/2,
                                          GetInterpolatedXPosition(), GetInterpolatedYPosition(),
                                          telegraphScaling, telegraphScaling,
                                          ship_move_angle + telegraphRotationOffset, 0);
            //std::cout << GetCurrentFrame();
        }
    }

}
