#pragma once

#include <map>
#include <utility>

#include "score.h"

struct ShipIndex
{
    static const int PC_CORE_NUM_FRAMES = 4;

    static const int SUBSHIP_NUM_FRAMES = 4;

    static constexpr float PC_AUTOPILOT_MOVESPEED = 16.0; // Don't want to use the rather high move speed for dramatic moments.
    static constexpr float PC_STANDARD_MOVESPEED = 16.0;

    static const int TRUE_DAMAGE_GRACE_TICKS_MAX = Timer::LPS*3.0;
    static const int COSMETIC_DAMAGE_GRACE_TICKS_MAX = Timer::LPS*0.4;

    static constexpr float PC_DAMAGE_BOUNCE_VELOCITY = 12.0;
    static constexpr float NPC_DAMAGE_BOUNCE_VELOCITY = 4.8;

    static const int NUM_CORE_OVERLOAD_FRAMES = 3;
    static const int NUM_CORE_OVERLOAD_FRAME_TICKS = 2;

    static const int DEATH_SEQUENCE_TICKS_DEFAULT = Timer::LPS*0.5;

    enum enumSubshipHullTypes
    {
        SUBSHIP_HULL_DORU = 0,
        SUBSHIP_HULL_XIPHOS = 1
    };
    static const int SUBSHIP_HULL_MARKER_BEGIN = SUBSHIP_HULL_DORU;
    static const int SUBSHIP_HULL_MARKER_END = SUBSHIP_HULL_XIPHOS;
    static const int SUBSHIP_HULL_NUM_TYPES = SUBSHIP_HULL_MARKER_END + 1;

    enum enumSubshipAttachments
    {
        ATTACH_DORU_NEUTRAL = 0,

        ATTACH_XIPHOS_NEUTRAL = 1,

        ATTACH_XIPHOS_VERTICAL_CROSS = 2,
        ATTACH_XIPHOS_VERTICAL_SLASH = 3,

        ATTACH_XIPHOS_HORIZONTAL_CROSS = 4,
        ATTACH_XIPHOS_HORIZONTAL_SLASH = 5
    };

    static inline const std::map<int, int>subshipNeutralAttachmentType
    {
        {SUBSHIP_HULL_DORU,   ATTACH_DORU_NEUTRAL},
        {SUBSHIP_HULL_XIPHOS, ATTACH_XIPHOS_NEUTRAL}
    };

    static inline const std::map<int, float>subshipFormationRotationOffset
    {
        {ATTACH_DORU_NEUTRAL,             0.0},
        {ATTACH_XIPHOS_NEUTRAL,           0.0},

        {ATTACH_XIPHOS_VERTICAL_CROSS,    ALLEGRO_PI},//2*ALLEGRO_PI*0.25}, // No change
        {ATTACH_XIPHOS_VERTICAL_SLASH,    0.0},
        {ATTACH_XIPHOS_HORIZONTAL_CROSS,  0.0}, //ALLEGRO_PI},
        {ATTACH_XIPHOS_HORIZONTAL_SLASH,  ALLEGRO_PI}
    };

    static inline const std::map<int, std::pair<float,float>>subshipAttachmentOffset // Lefthand is assumed, and symmetrical.
    {
        {ATTACH_DORU_NEUTRAL,            {-128,  96}},

        {ATTACH_XIPHOS_NEUTRAL,          { -64,   0}},
        {ATTACH_XIPHOS_VERTICAL_CROSS,   {  16, -72}},
        {ATTACH_XIPHOS_VERTICAL_SLASH,   {  56,  72}},
        {ATTACH_XIPHOS_HORIZONTAL_CROSS, {  88,  56}},
        {ATTACH_XIPHOS_HORIZONTAL_SLASH, { -40, -56}}
    };


    enum enumNPCHullTypes
    {
        // Small, fish shaped ships
        HULL_NPC_RAY = 0,
        HULL_NPC_OCELLUS = 1,
        HULL_NPC_ANGELFISH = 2,
        HULL_NPC_ANTLION = 3,

        // Drones
        HULL_NPC_DISCUS = 4,
        HULL_NPC_CONSTANT = 5,
        HULL_NPC_CROSS = 6,
        HULL_NPC_MISSILE = 7,

        // Big ships
        HULL_NPC_PARTISAN = 8,
        HULL_NPC_SUNFISH = 9,
        HULL_NPC_SPIRIT = 10,
        HULL_NPC_AEGIS = 11,

        // Mecha
        HULL_NPC_LUNARIAN = 12,  // Similar to the Lunarian from Krampus 2023. Uses a recharging energy shield, teleportation, sword.

        // Boss ship
        HULL_NPC_SCHEHERAZADE = 13
    };
    static const int HULL_NPC_SMALL_SHIP_MARKER_BEGIN = HULL_NPC_RAY;
    static const int HULL_NPC_SMALL_SHIP_MARKER_END = HULL_NPC_ANTLION;
    static const int HULL_NPC_DRONE_MARKER_BEGIN = HULL_NPC_DISCUS;
    static const int HULL_NPC_DRONE_MARKER_END = HULL_NPC_MISSILE;
    static const int HULL_NPC_LARGE_SHIP_MARKER_BEGIN = HULL_NPC_PARTISAN;
    static const int HULL_NPC_LARGE_SHIP_MARKER_END = HULL_NPC_AEGIS;
    static const int HULL_NPC_MECHA_MARKER_BEGIN = HULL_NPC_LUNARIAN;
    static const int HULL_NPC_MECHA_MARKER_END = HULL_NPC_LUNARIAN;
    static const int HULL_NPC_BOSS_MARKER_BEGIN = HULL_NPC_SCHEHERAZADE;
    static const int HULL_NPC_BOSS_MARKER_END = HULL_NPC_SCHEHERAZADE;
    static const int HULL_NPC_NUM_TYPES = HULL_NPC_BOSS_MARKER_END + 1;

    static const inline std::map<int, std::pair<float,float>> npcSpriteDimensions
    {
        {HULL_NPC_RAY,          {128,  128}},
	    {HULL_NPC_OCELLUS,      {128,  128}},
	    {HULL_NPC_ANGELFISH,    {128,  128}},
        {HULL_NPC_ANTLION,      {128,  128}},
        {HULL_NPC_DISCUS,       {128,  128}},
        {HULL_NPC_CONSTANT,     {128,  128}},
        {HULL_NPC_CROSS,        {128,  128}},
        {HULL_NPC_MISSILE,      {128,  128}},
        {HULL_NPC_PARTISAN,     {256,  256}},
        {HULL_NPC_SUNFISH,      {256,  256}},
        {HULL_NPC_SPIRIT,       {256,  256}},
        {HULL_NPC_AEGIS,        {256,  256}},
        {HULL_NPC_LUNARIAN,     {128,  128}},
        {HULL_NPC_SCHEHERAZADE, {512,  512}}
    };

    static const inline std::map<int, std::pair<float,float>> npcHitboxDimensions
    {
        {HULL_NPC_RAY,          {64,64}},
	    {HULL_NPC_OCELLUS,      {64,64}},
	    {HULL_NPC_ANGELFISH,    {64,64}},
        {HULL_NPC_ANTLION,      {64,64}},
        {HULL_NPC_DISCUS,       {64,64}},
        {HULL_NPC_CONSTANT,     {64,64}},
        {HULL_NPC_CROSS,        {64,64}},
        {HULL_NPC_MISSILE,      {64,64}},
        {HULL_NPC_PARTISAN,     {128,128}},
        {HULL_NPC_SUNFISH,      {128,128}},
        {HULL_NPC_SPIRIT,       {128,128}},
        {HULL_NPC_AEGIS,        {128,128}},
        {HULL_NPC_LUNARIAN,     {64,64}},
        {HULL_NPC_SCHEHERAZADE, {128,128}}
    };

    static const inline std::map<int, std::pair<float,float>> npcHitboxXYOffset
    {
        {HULL_NPC_RAY,          {-32,-32}},
	    {HULL_NPC_OCELLUS,      {-32,-32}},
	    {HULL_NPC_ANGELFISH,    {-32,-32}},
        {HULL_NPC_ANTLION,      {-32,-32}},
        {HULL_NPC_DISCUS,       {-32,-32}},
        {HULL_NPC_CONSTANT,     {-32,-32}},
        {HULL_NPC_CROSS,        {-32,-32}},
        {HULL_NPC_MISSILE,      {-32,-32}},
        {HULL_NPC_PARTISAN,     {-64,-64}},
        {HULL_NPC_SUNFISH,      {-64,-64}},
        {HULL_NPC_SPIRIT,       {-64,-64}},
        {HULL_NPC_AEGIS,        {-64,-64}},
        {HULL_NPC_LUNARIAN,     {-32,-32}},
        {HULL_NPC_SCHEHERAZADE, {-64,-64}}
    };

    static const inline std::map<int, int> npcScoreTier
	{
	    {HULL_NPC_RAY, Score::SCORE_TIER_SMALL_SHIP},
	    {HULL_NPC_OCELLUS, Score::SCORE_TIER_SMALL_SHIP},
	    {HULL_NPC_ANGELFISH, Score::SCORE_TIER_SMALL_SHIP},
        {HULL_NPC_ANTLION, Score::SCORE_TIER_SMALL_SHIP},
        {HULL_NPC_DISCUS, Score::SCORE_TIER_POPCORN},
        {HULL_NPC_CONSTANT, Score::SCORE_TIER_DRONE},
        {HULL_NPC_CROSS, Score::SCORE_TIER_DRONE},
        {HULL_NPC_MISSILE, Score::SCORE_TIER_DRONE},
        {HULL_NPC_PARTISAN, Score::SCORE_TIER_LARGE_SHIP},
        {HULL_NPC_SUNFISH, Score::SCORE_TIER_LARGE_SHIP},
        {HULL_NPC_SPIRIT, Score::SCORE_TIER_LARGE_SHIP},
        {HULL_NPC_AEGIS, Score::SCORE_TIER_LARGE_SHIP},
        {HULL_NPC_LUNARIAN, Score::SCORE_TIER_MECHA},
        {HULL_NPC_SCHEHERAZADE, Score::SCORE_TIER_BOSS}
	};

	static constexpr float DEFAULT_NPC_MAX_HP = 20;
	static const inline std::map<int, float> npcMaxHP
	{
	    {HULL_NPC_RAY,          DEFAULT_NPC_MAX_HP},
	    {HULL_NPC_OCELLUS,      DEFAULT_NPC_MAX_HP},
	    {HULL_NPC_ANGELFISH,    DEFAULT_NPC_MAX_HP},
        {HULL_NPC_ANTLION,      DEFAULT_NPC_MAX_HP},
        {HULL_NPC_DISCUS,       DEFAULT_NPC_MAX_HP*0.25},
        {HULL_NPC_CONSTANT,     DEFAULT_NPC_MAX_HP*0.33},
        {HULL_NPC_CROSS,        DEFAULT_NPC_MAX_HP*0.7},
        {HULL_NPC_MISSILE,      DEFAULT_NPC_MAX_HP*0.20},
        {HULL_NPC_PARTISAN,     DEFAULT_NPC_MAX_HP*2.0},
        {HULL_NPC_SUNFISH,      DEFAULT_NPC_MAX_HP*3.0},
        {HULL_NPC_SPIRIT,       DEFAULT_NPC_MAX_HP*1.2},
        {HULL_NPC_AEGIS,        DEFAULT_NPC_MAX_HP*4.0},
        {HULL_NPC_LUNARIAN,     DEFAULT_NPC_MAX_HP},
        {HULL_NPC_SCHEHERAZADE, DEFAULT_NPC_MAX_HP*40},
	};

	static constexpr float DEFAULT_NPC_MOVE_SPEED = 2.0;
	static const inline std::map<int, float> npcMoveSpeed
	{
	    {HULL_NPC_RAY,          DEFAULT_NPC_MOVE_SPEED},
	    {HULL_NPC_OCELLUS,      DEFAULT_NPC_MOVE_SPEED},
	    {HULL_NPC_ANGELFISH,    DEFAULT_NPC_MOVE_SPEED},
        {HULL_NPC_ANTLION,      DEFAULT_NPC_MOVE_SPEED},
        {HULL_NPC_DISCUS,       DEFAULT_NPC_MOVE_SPEED*2.0},
        {HULL_NPC_CONSTANT,     DEFAULT_NPC_MOVE_SPEED*3.0},
        {HULL_NPC_CROSS,        DEFAULT_NPC_MOVE_SPEED*0.7},
        {HULL_NPC_MISSILE,      DEFAULT_NPC_MOVE_SPEED*6.0},
        {HULL_NPC_PARTISAN,     DEFAULT_NPC_MOVE_SPEED*1.5},
        {HULL_NPC_SUNFISH,      DEFAULT_NPC_MOVE_SPEED*0.75},
        {HULL_NPC_SPIRIT,       DEFAULT_NPC_MOVE_SPEED*4.0},
        {HULL_NPC_AEGIS,        DEFAULT_NPC_MOVE_SPEED*0.5},
        {HULL_NPC_LUNARIAN,     DEFAULT_NPC_MOVE_SPEED*3.0},
        {HULL_NPC_SCHEHERAZADE, DEFAULT_NPC_MOVE_SPEED*8.0},
	};
};

