

#include <AllegroFlare/Screens/TitleScreen.hpp>

#include <AllegroFlare/ColorKit.hpp>
#include <AllegroFlare/EventNames.hpp>
#include <AllegroFlare/MotionKit.hpp>
#include <AllegroFlare/Placement2D.hpp>
#include <AllegroFlare/VirtualControllers/GenericController.hpp>
#include <algorithm>
#include <allegro5/allegro_primitives.h>
#include <cmath>
#include <iostream>
#include <set>
#include <sstream>
#include <stdexcept>


namespace AllegroFlare
{
namespace Screens
{


TitleScreen::TitleScreen(AllegroFlare::EventEmitter* event_emitter, AllegroFlare::FontBin* font_bin, AllegroFlare::BitmapBin* bitmap_bin, std::size_t surface_width, std::size_t surface_height, std::string title_text, std::string copyright_text, std::string title_bitmap_name, std::string title_font_name, std::string menu_font_name, std::string copyright_font_name, ALLEGRO_COLOR title_text_color, ALLEGRO_COLOR menu_text_color, ALLEGRO_COLOR menu_selector_color, ALLEGRO_COLOR menu_selector_outline_color, ALLEGRO_COLOR menu_selected_text_color, ALLEGRO_COLOR copyright_text_color, float menu_selector_outline_stroke_thickness, int title_font_size, int menu_font_size, int copyright_font_size)
   : AllegroFlare::Screens::Base("TitleScreen")
   , event_emitter(event_emitter)
   , font_bin(font_bin)
   , bitmap_bin(bitmap_bin)
   , surface_width(surface_width)
   , surface_height(surface_height)
   , title_text(title_text)
   , copyright_text(copyright_text)
   , title_bitmap_name(title_bitmap_name)
   , title_font_name(title_font_name)
   , menu_font_name(menu_font_name)
   , copyright_font_name(copyright_font_name)
   , title_text_color(title_text_color)
   , menu_text_color(menu_text_color)
   , menu_selector_color(menu_selector_color)
   , menu_selector_outline_color(menu_selector_outline_color)
   , menu_selected_text_color(menu_selected_text_color)
   , copyright_text_color(copyright_text_color)
   , menu_selector_outline_stroke_thickness(menu_selector_outline_stroke_thickness)
   , title_font_size(title_font_size)
   , menu_font_size(menu_font_size)
   , copyright_font_size(copyright_font_size)
   , menu_options(build_default_menu_options())
   , on_menu_choice_callback_func()
   , on_menu_choice_callback_func_user_data(nullptr)
   , on_finished_callback_func()
   , on_finished_callback_func_user_data(nullptr)
   , title_position_x(1920 / 2)
   , title_position_y((1080 / 24 * 9))
   , menu_position_x(1920 / 2)
   , menu_position_y(1080 / 12 * 7)
   , cursor_position(0)
   , menu_move_sound_effect_identifier("menu_move")
   , menu_move_sound_effect_enabled(true)
   , menu_select_option_sound_effect_identifier("menu_select")
   , menu_select_option_sound_effect_enabled(true)
   , menu_option_selection_to_activation_delay(1.0f)
   , reveal_duration(1.0f)
   , reveal_started_at(0.0f)
   , showing_menu(false)
   , showing_copyright(false)
   , state(STATE_UNDEF)
   , state_is_busy(false)
   , state_changed_at(0.0f)
   , menu_option_chosen(false)
   , menu_option_chosen_at(0.0f)
   , menu_option_activated(false)
   , showing_confirmation_dialog(false)
{
}


TitleScreen::~TitleScreen()
{
}


void TitleScreen::set_event_emitter(AllegroFlare::EventEmitter* event_emitter)
{
   this->event_emitter = event_emitter;
}


void TitleScreen::set_font_bin(AllegroFlare::FontBin* font_bin)
{
   this->font_bin = font_bin;
}


void TitleScreen::set_bitmap_bin(AllegroFlare::BitmapBin* bitmap_bin)
{
   this->bitmap_bin = bitmap_bin;
}


void TitleScreen::set_surface_width(std::size_t surface_width)
{
   this->surface_width = surface_width;
}


void TitleScreen::set_surface_height(std::size_t surface_height)
{
   this->surface_height = surface_height;
}


void TitleScreen::set_title_text(std::string title_text)
{
   this->title_text = title_text;
}


void TitleScreen::set_copyright_text(std::string copyright_text)
{
   this->copyright_text = copyright_text;
}


void TitleScreen::set_title_bitmap_name(std::string title_bitmap_name)
{
   this->title_bitmap_name = title_bitmap_name;
}


void TitleScreen::set_title_font_name(std::string title_font_name)
{
   this->title_font_name = title_font_name;
}


void TitleScreen::set_menu_font_name(std::string menu_font_name)
{
   this->menu_font_name = menu_font_name;
}


void TitleScreen::set_copyright_font_name(std::string copyright_font_name)
{
   this->copyright_font_name = copyright_font_name;
}


void TitleScreen::set_title_text_color(ALLEGRO_COLOR title_text_color)
{
   this->title_text_color = title_text_color;
}


void TitleScreen::set_menu_text_color(ALLEGRO_COLOR menu_text_color)
{
   this->menu_text_color = menu_text_color;
}


void TitleScreen::set_menu_selector_color(ALLEGRO_COLOR menu_selector_color)
{
   this->menu_selector_color = menu_selector_color;
}


void TitleScreen::set_menu_selector_outline_color(ALLEGRO_COLOR menu_selector_outline_color)
{
   this->menu_selector_outline_color = menu_selector_outline_color;
}


void TitleScreen::set_menu_selected_text_color(ALLEGRO_COLOR menu_selected_text_color)
{
   this->menu_selected_text_color = menu_selected_text_color;
}


void TitleScreen::set_copyright_text_color(ALLEGRO_COLOR copyright_text_color)
{
   this->copyright_text_color = copyright_text_color;
}


void TitleScreen::set_menu_selector_outline_stroke_thickness(float menu_selector_outline_stroke_thickness)
{
   this->menu_selector_outline_stroke_thickness = menu_selector_outline_stroke_thickness;
}


void TitleScreen::set_title_font_size(int title_font_size)
{
   this->title_font_size = title_font_size;
}


void TitleScreen::set_menu_font_size(int menu_font_size)
{
   this->menu_font_size = menu_font_size;
}


void TitleScreen::set_copyright_font_size(int copyright_font_size)
{
   this->copyright_font_size = copyright_font_size;
}


void TitleScreen::set_on_menu_choice_callback_func(std::function<void(AllegroFlare::Screens::TitleScreen*, void*)> on_menu_choice_callback_func)
{
   this->on_menu_choice_callback_func = on_menu_choice_callback_func;
}


void TitleScreen::set_on_menu_choice_callback_func_user_data(void* on_menu_choice_callback_func_user_data)
{
   this->on_menu_choice_callback_func_user_data = on_menu_choice_callback_func_user_data;
}


void TitleScreen::set_on_finished_callback_func(std::function<void(AllegroFlare::Screens::TitleScreen*, void*)> on_finished_callback_func)
{
   this->on_finished_callback_func = on_finished_callback_func;
}


void TitleScreen::set_on_finished_callback_func_user_data(void* on_finished_callback_func_user_data)
{
   this->on_finished_callback_func_user_data = on_finished_callback_func_user_data;
}


void TitleScreen::set_title_position_x(float title_position_x)
{
   this->title_position_x = title_position_x;
}


void TitleScreen::set_title_position_y(float title_position_y)
{
   this->title_position_y = title_position_y;
}


void TitleScreen::set_menu_position_x(float menu_position_x)
{
   this->menu_position_x = menu_position_x;
}


void TitleScreen::set_menu_position_y(float menu_position_y)
{
   this->menu_position_y = menu_position_y;
}


void TitleScreen::set_menu_move_sound_effect_identifier(std::string menu_move_sound_effect_identifier)
{
   this->menu_move_sound_effect_identifier = menu_move_sound_effect_identifier;
}


void TitleScreen::set_menu_move_sound_effect_enabled(bool menu_move_sound_effect_enabled)
{
   this->menu_move_sound_effect_enabled = menu_move_sound_effect_enabled;
}


void TitleScreen::set_menu_select_option_sound_effect_identifier(std::string menu_select_option_sound_effect_identifier)
{
   this->menu_select_option_sound_effect_identifier = menu_select_option_sound_effect_identifier;
}


void TitleScreen::set_menu_select_option_sound_effect_enabled(bool menu_select_option_sound_effect_enabled)
{
   this->menu_select_option_sound_effect_enabled = menu_select_option_sound_effect_enabled;
}


void TitleScreen::set_menu_option_selection_to_activation_delay(float menu_option_selection_to_activation_delay)
{
   this->menu_option_selection_to_activation_delay = menu_option_selection_to_activation_delay;
}


void TitleScreen::set_reveal_duration(float reveal_duration)
{
   this->reveal_duration = reveal_duration;
}


std::size_t TitleScreen::get_surface_width() const
{
   return surface_width;
}


std::size_t TitleScreen::get_surface_height() const
{
   return surface_height;
}


std::string TitleScreen::get_title_text() const
{
   return title_text;
}


std::string TitleScreen::get_copyright_text() const
{
   return copyright_text;
}


std::string TitleScreen::get_title_bitmap_name() const
{
   return title_bitmap_name;
}


std::string TitleScreen::get_title_font_name() const
{
   return title_font_name;
}


std::string TitleScreen::get_menu_font_name() const
{
   return menu_font_name;
}


std::string TitleScreen::get_copyright_font_name() const
{
   return copyright_font_name;
}


ALLEGRO_COLOR TitleScreen::get_title_text_color() const
{
   return title_text_color;
}


ALLEGRO_COLOR TitleScreen::get_menu_text_color() const
{
   return menu_text_color;
}


ALLEGRO_COLOR TitleScreen::get_menu_selector_color() const
{
   return menu_selector_color;
}


ALLEGRO_COLOR TitleScreen::get_menu_selector_outline_color() const
{
   return menu_selector_outline_color;
}


ALLEGRO_COLOR TitleScreen::get_menu_selected_text_color() const
{
   return menu_selected_text_color;
}


ALLEGRO_COLOR TitleScreen::get_copyright_text_color() const
{
   return copyright_text_color;
}


float TitleScreen::get_menu_selector_outline_stroke_thickness() const
{
   return menu_selector_outline_stroke_thickness;
}


int TitleScreen::get_title_font_size() const
{
   return title_font_size;
}


int TitleScreen::get_menu_font_size() const
{
   return menu_font_size;
}


int TitleScreen::get_copyright_font_size() const
{
   return copyright_font_size;
}


std::vector<std::pair<std::string, std::string>> TitleScreen::get_menu_options() const
{
   return menu_options;
}


std::function<void(AllegroFlare::Screens::TitleScreen*, void*)> TitleScreen::get_on_menu_choice_callback_func() const
{
   return on_menu_choice_callback_func;
}


void* TitleScreen::get_on_menu_choice_callback_func_user_data() const
{
   return on_menu_choice_callback_func_user_data;
}


std::function<void(AllegroFlare::Screens::TitleScreen*, void*)> TitleScreen::get_on_finished_callback_func() const
{
   return on_finished_callback_func;
}


void* TitleScreen::get_on_finished_callback_func_user_data() const
{
   return on_finished_callback_func_user_data;
}


float TitleScreen::get_title_position_x() const
{
   return title_position_x;
}


float TitleScreen::get_title_position_y() const
{
   return title_position_y;
}


float TitleScreen::get_menu_position_x() const
{
   return menu_position_x;
}


float TitleScreen::get_menu_position_y() const
{
   return menu_position_y;
}


int TitleScreen::get_cursor_position() const
{
   return cursor_position;
}


std::string TitleScreen::get_menu_move_sound_effect_identifier() const
{
   return menu_move_sound_effect_identifier;
}


bool TitleScreen::get_menu_move_sound_effect_enabled() const
{
   return menu_move_sound_effect_enabled;
}


std::string TitleScreen::get_menu_select_option_sound_effect_identifier() const
{
   return menu_select_option_sound_effect_identifier;
}


bool TitleScreen::get_menu_select_option_sound_effect_enabled() const
{
   return menu_select_option_sound_effect_enabled;
}


float TitleScreen::get_menu_option_selection_to_activation_delay() const
{
   return menu_option_selection_to_activation_delay;
}


float TitleScreen::get_reveal_duration() const
{
   return reveal_duration;
}


void TitleScreen::TODO()
{
   // There is some confusing naming between "selected", "chosen", etc.  Selected seems to signify
   // either "highlighted" or "chosen" depending on the casses.
   // Setup the "on_finished", indicating the title screen has "timed out" and may want to loop back around
     // to logos or something
   return;
}

void TitleScreen::set_font_name(std::string font_name)
{
   title_font_name = font_name;
   menu_font_name = font_name;
   copyright_font_name = font_name;
   return;
}

void TitleScreen::set_state(uint32_t state, bool override_if_busy)
{
   if (!(is_valid_state(state)))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::set_state]: error: guard \"is_valid_state(state)\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::set_state: error: guard \"is_valid_state(state)\" not met");
   }
   if (this->state == state) return;
   if (!override_if_busy && state_is_busy) return;
   uint32_t previous_state = this->state;

   switch (state)
   {
      case STATE_REVEALING:
         reveal_started_at = al_get_time(); // TODO: Consider injecting time
         cursor_position = 0;
         showing_menu = false;
         showing_copyright = false;
         menu_option_chosen = false;
         menu_option_activated = false;
         menu_option_chosen_at = 0.0f;
      break;

      case STATE_AWAITING_USER_INPUT:
         menu_option_chosen = false;
         menu_option_activated = false;
         show_menu();
      break;

      case STATE_MENU_OPTION_IS_CHOSEN:
         menu_option_chosen = true;
         menu_option_activated = false;
         menu_option_chosen_at = al_get_time();
      break;

      case STATE_AWAITING_USER_CONFIRMATION:
         showing_confirmation_dialog = true;
      break;

      case STATE_FINISHED:
      break;

      default:
         throw std::runtime_error("weird error");
      break;
   }

   this->state = state;
   state_changed_at = al_get_time();

   return;
}

void TitleScreen::update(float time_now)
{
   if (!(is_valid_state(state)))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::update]: error: guard \"is_valid_state(state)\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::update: error: guard \"is_valid_state(state)\" not met");
   }
   float state_age = infer_age(state_changed_at, time_now);

   switch (state)
   {
      case STATE_REVEALING: {
         float reveal_age = infer_reveal_age();
         if (state_age > reveal_duration)
         {
            set_state(STATE_AWAITING_USER_INPUT);
         }
      } break;

      case STATE_AWAITING_USER_INPUT:
      break;

      case STATE_MENU_OPTION_IS_CHOSEN:
         if (!menu_option_activated && state_age > menu_option_selection_to_activation_delay)
         {
            //std::string current_menu_option_value = infer_current_menu_option_value();
            activate_current_selected_menu_option(); //current_menu_option_value);
            menu_option_chosen = false;
            menu_option_activated = true;
            set_state(STATE_FINISHED);
         }
      break;

      case STATE_AWAITING_USER_CONFIRMATION:
         // TODO
      break;

      case STATE_FINISHED:
      break;

      default:
         throw std::runtime_error("weird error");
      break;
   }

   return;
}

bool TitleScreen::is_valid_state(uint32_t state)
{
   std::set<uint32_t> valid_states =
   {
      STATE_REVEALING,
      STATE_AWAITING_USER_INPUT,
      STATE_MENU_OPTION_IS_CHOSEN,
      STATE_AWAITING_USER_CONFIRMATION,
      STATE_FINISHED,
   };
   return (valid_states.count(state) > 0);
}

float TitleScreen::infer_age(float time_of_event, float time_now)
{
   return std::max(0.0f, time_now - time_of_event);
}

float TitleScreen::infer_reveal_age(float time_now)
{
   return std::max(0.0f, infer_age(reveal_started_at, time_now));
}

float TitleScreen::infer_reveal_age_n(float time_now)
{
   if (!((reveal_duration != 0.0f)))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::infer_reveal_age_n]: error: guard \"(reveal_duration != 0.0f)\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::infer_reveal_age_n: error: guard \"(reveal_duration != 0.0f)\" not met");
   }
   return std::max(0.0f, std::min(1.0f, infer_reveal_age(time_now) / reveal_duration));
}

void TitleScreen::show_menu()
{
   showing_menu = true;
   showing_copyright = true;
   return;
}

void TitleScreen::on_activate()
{
   set_state(STATE_REVEALING);
   return;
}

void TitleScreen::skip_to_title()
{
   set_state(STATE_AWAITING_USER_INPUT);
   return;
}

void TitleScreen::set_menu_options(std::vector<std::pair<std::string, std::string>> menu_options)
{
   this->menu_options = menu_options;
   cursor_position = 0;
   return;
}

void TitleScreen::move_cursor_up()
{
   if (!processing_user_input_on_main_menu()) return;

   if (menu_move_sound_effect_enabled) play_menu_move_sound_effect();

   cursor_position--;
   if (cursor_position < 0) cursor_position += menu_options.size();

   return;
}

void TitleScreen::move_cursor_down()
{
   if (!processing_user_input_on_main_menu()) return;

   if (menu_move_sound_effect_enabled) play_menu_move_sound_effect();

   cursor_position++;
   if (cursor_position >= menu_options.size()) cursor_position = cursor_position % menu_options.size();

   return;
}

void TitleScreen::activate_current_selected_menu_option()
{
   if (!(event_emitter))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::activate_current_selected_menu_option]: error: guard \"event_emitter\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::activate_current_selected_menu_option: error: guard \"event_emitter\" not met");
   }
   // TODO: Consider case where there is an emtpy list
   // TODO: Remove the emit_game_event as a technique here, rely on callback only
   std::string current_menu_option_value = infer_current_menu_option_value();
   event_emitter->emit_game_event(current_menu_option_value);
   // TODO: Test this callback
   if (on_menu_choice_callback_func) on_menu_choice_callback_func(this, on_menu_choice_callback_func_user_data);
   return;
}

void TitleScreen::select_menu_option()
{
   if (!processing_user_input_on_main_menu()) return;

   if (menu_is_empty())
   {
      std::cout <<
         "[AllegroFlare::Screens::TitleScreen::select_menu_option()] error: can not select a menu item, "
         "the menu is empty."
         << std::endl;
      return;
   }

   if (current_menu_option_must_be_confirmed()) set_state(STATE_AWAITING_USER_CONFIRMATION);
   else set_state(STATE_MENU_OPTION_IS_CHOSEN);

   if (menu_select_option_sound_effect_enabled) play_menu_select_option_sound_effect();

   return;
}

bool TitleScreen::current_menu_option_must_be_confirmed()
{
   std::set<std::string> choices_requiring_confirmation = { "exit_game*" };
   std::string current_menu_value = infer_current_menu_option_value();
   return choices_requiring_confirmation.find(current_menu_value) != choices_requiring_confirmation.end();
}

bool TitleScreen::processing_user_input_on_main_menu()
{
   return is_state(STATE_AWAITING_USER_INPUT); 
}

bool TitleScreen::is_state(uint32_t possible_state)
{
   return (state == possible_state);
}

void TitleScreen::primary_timer_func()
{
   update();
   render();
   return;
}

void TitleScreen::render()
{
   if (!(al_is_system_installed()))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::render]: error: guard \"al_is_system_installed()\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::render: error: guard \"al_is_system_installed()\" not met");
   }
   if (!(al_is_primitives_addon_initialized()))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::render]: error: guard \"al_is_primitives_addon_initialized()\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::render: error: guard \"al_is_primitives_addon_initialized()\" not met");
   }
   if (!(al_is_font_addon_initialized()))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::render]: error: guard \"al_is_font_addon_initialized()\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::render: error: guard \"al_is_font_addon_initialized()\" not met");
   }
   draw_title();
   if (showing_copyright) draw_copyright_text();
   if (showing_menu) draw_menu();
   if (showing_confirmation_dialog) draw_confirmation_dialog();
   return;
}

void TitleScreen::draw_title()
{
   if (!title_bitmap_name.empty())
   {
      ALLEGRO_BITMAP *title_bitmap = obtain_title_bitmap();
      float reveal_opacity = infer_reveal_age_n();
      ALLEGRO_COLOR tint = ALLEGRO_COLOR{reveal_opacity, reveal_opacity, reveal_opacity, reveal_opacity};
      if (title_bitmap)
      {
         AllegroFlare::Placement2D place;
         place.position.x = title_position_x;
         place.position.y = title_position_y;
         place.size.x = al_get_bitmap_width(title_bitmap);
         place.size.y = al_get_bitmap_height(title_bitmap);
         place.start_transform();
         al_draw_tinted_bitmap(title_bitmap, tint, 0, 0, 0);
         place.restore_transform();
      }
   }
   else if (!title_text.empty())
   {
      // TODO: review guards on this function
      ALLEGRO_FONT *title_font = obtain_title_font();
      float reveal_opacity = infer_reveal_age_n();
      ALLEGRO_COLOR revealed_color = ALLEGRO_COLOR{
         reveal_opacity * title_text_color.r,
         reveal_opacity * title_text_color.g,
         reveal_opacity * title_text_color.b,
         reveal_opacity * title_text_color.a
      };
      //int surface_width = 1920;
      //int surface_height = 1080;
      al_draw_text(
         title_font,
         revealed_color,
         title_position_x,
         title_position_y,
         ALLEGRO_ALIGN_CENTER,
         get_title_text().c_str()
      );
   }
   return;
}

void TitleScreen::draw_copyright_text()
{
   // TODO: review guards on this function
   ALLEGRO_FONT *copyright_font = obtain_copyright_font();
   //int surface_width = 1920;
   //int surface_height = 1080;
   float line_height = al_get_font_line_height(copyright_font);

   int number_of_lines = 1;
   for (int i = 0; i < copyright_text.size(); i++) if (copyright_text[i] == '\n') number_of_lines++;

   al_draw_multiline_text(
      copyright_font,
      copyright_text_color,
      surface_width/2,
      surface_height - 80 - (int)(number_of_lines * line_height / 2),
      surface_width * 2,
      line_height,
      ALLEGRO_ALIGN_CENTER,
      copyright_text.c_str()
   );

   return;
}

void TitleScreen::draw_cursor_box(float x, float y, float width, float height, ALLEGRO_COLOR fill_color, ALLEGRO_COLOR outline_color, float outline_stroke_thickness, bool menu_option_chosen, float menu_option_chosen_at, float menu_option_selection_to_activation_delay, float time_now)
{
   ALLEGRO_COLOR result_fill_color = fill_color; //ALLEGRO_COLOR{0, 0, 0, 0};
   ALLEGRO_COLOR result_outline_color = outline_color; //ALLEGRO_COLOR{1, 1, 1, 1};

   if (menu_option_chosen)
   {
      float selection_animation_length = menu_option_selection_to_activation_delay;
      float selection_strobes_per_second = 14.0f;

      float menu_option_chosen_at_age = AllegroFlare::MotionKit::age(menu_option_chosen_at, time_now);
      bool strobe_on = AllegroFlare::MotionKit::strobe(
         menu_option_chosen_at,
         time_now,
         selection_strobes_per_second
      );

      float menu_option_chosen_at_normalized_age = AllegroFlare::MotionKit::normalize_age(
         menu_option_chosen_at,
         menu_option_chosen_at + selection_animation_length,
         time_now
      );
      ALLEGRO_COLOR cursor_color_a = AllegroFlare::ColorKit::fade(result_fill_color, 1.0);
      ALLEGRO_COLOR cursor_color_b = AllegroFlare::ColorKit::fade(result_fill_color, 0.5);
      ALLEGRO_COLOR cursor_animation_at_rest_color = AllegroFlare::ColorKit::fade(result_fill_color, 0.2);

      ALLEGRO_COLOR strobing_color = AllegroFlare::ColorKit::mix(
         cursor_color_a,
         cursor_color_b,
         strobe_on ? 1.0f : 0.0f
      );

      result_fill_color = AllegroFlare::ColorKit::mix(
         strobing_color,
         cursor_animation_at_rest_color,
         menu_option_chosen_at_normalized_age
      );
   }

   float h_box_width = width * 0.5;
   float h_box_height = height * 0.5;

   // draw the fill
   al_draw_filled_rectangle(x-h_box_width, y-h_box_height, x+h_box_width, y+h_box_height, result_fill_color);

   // draw the outline (which is invisible by default)
   al_draw_rectangle(
      x-h_box_width,
      y-h_box_height,
      x+h_box_width,
      y+h_box_height,
      result_outline_color,
      outline_stroke_thickness
   );
   return;
}

void TitleScreen::draw_menu()
{
   if (!(al_is_primitives_addon_initialized()))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::draw_menu]: error: guard \"al_is_primitives_addon_initialized()\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::draw_menu: error: guard \"al_is_primitives_addon_initialized()\" not met");
   }
   // TODO: review guards on this function
   ALLEGRO_FONT *menu_font = obtain_menu_font();
   //int surface_width = 1920;
   //int surface_height = 1080;
   float h_font_line_height = (int)(al_get_font_line_height(menu_font) * 0.5f);
   float menu_item_vertical_spacing = (int)(al_get_font_line_height(menu_font) * 1.4f);
   int menu_item_num = 0;

   // get longest menu option text length
   int longest_menu_option_text_width = 0;
   for (auto &menu_option : menu_options)
   {
      std::string menu_item_text = std::get<0>(menu_option);
      int this_menu_item_text_width = al_get_text_width(menu_font, menu_item_text.c_str());
      if (this_menu_item_text_width > longest_menu_option_text_width)
         longest_menu_option_text_width = this_menu_item_text_width;
   }

   // render each menu item
   for (auto &menu_option : menu_options)
   {
      bool showing_cursor_on_this_option = false;
      if (menu_item_num == cursor_position) showing_cursor_on_this_option = true;
      std::string menu_item_text = std::get<0>(menu_option);

      ALLEGRO_COLOR this_menu_text_color = showing_cursor_on_this_option
         ? (menu_option_chosen ? menu_text_color : menu_selected_text_color) : menu_text_color;

      float x = menu_position_x;
      float y = menu_position_y + menu_item_vertical_spacing * menu_item_num;

      if (showing_cursor_on_this_option)
      {
         float box_width = longest_menu_option_text_width + 148;
         float box_height = al_get_font_line_height(menu_font) + 6;

         draw_cursor_box(
            x,
            y,
            box_width,
            box_height,
            menu_selector_color,
            menu_selector_outline_color,
            menu_selector_outline_stroke_thickness,
            menu_option_chosen,
            menu_option_chosen_at,
            menu_option_selection_to_activation_delay,
            al_get_time()
         );
      }

      al_draw_text(
         menu_font,
         this_menu_text_color,
         (int)x,
         (int)(y-h_font_line_height),
         ALLEGRO_ALIGN_CENTER,
         menu_item_text.c_str()
      );

      menu_item_num++;
   }
   return;
}

void TitleScreen::draw_confirmation_dialog()
{
   if (!(al_is_primitives_addon_initialized()))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::draw_confirmation_dialog]: error: guard \"al_is_primitives_addon_initialized()\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::draw_confirmation_dialog: error: guard \"al_is_primitives_addon_initialized()\" not met");
   }
   // TODO: review guards on this function
   ALLEGRO_FONT *menu_font = obtain_menu_font();
   //int surface_width = 1920;
   //int surface_height = 1080;
   float h_font_line_height = (int)(al_get_font_line_height(menu_font) * 0.5f);
   float menu_item_vertical_spacing = (int)(al_get_font_line_height(menu_font) * 1.4f);
   int menu_item_num = 0;

   // get longest menu option text length
   int longest_menu_option_text_width = 0;
   auto confirmation_dialog_menu_options = build_confirmation_dialog_menu_options();
   for (auto &menu_option : confirmation_dialog_menu_options)
   {
      std::string menu_item_text = std::get<0>(menu_option);
      int this_menu_item_text_width = al_get_text_width(menu_font, menu_item_text.c_str());
      if (this_menu_item_text_width > longest_menu_option_text_width)
         longest_menu_option_text_width = this_menu_item_text_width;
   }

   // render each menu item

   for (auto &menu_option : confirmation_dialog_menu_options)
   {
      bool showing_cursor_on_this_option = false;
      if (menu_item_num == cursor_position) showing_cursor_on_this_option = true;
      std::string menu_item_text = std::get<0>(menu_option);

      ALLEGRO_COLOR this_menu_text_color = showing_cursor_on_this_option
         ? (menu_option_chosen ? menu_text_color : menu_selected_text_color) : menu_text_color;

      float x = menu_position_x + 200;
      float y = menu_position_y + menu_item_vertical_spacing * menu_item_num;

      if (showing_cursor_on_this_option)
      {
         float box_width = longest_menu_option_text_width + 148;
         float box_height = al_get_font_line_height(menu_font) + 6;

         draw_cursor_box(
            x,
            y,
            box_width,
            box_height,
            menu_selector_color,
            menu_selector_outline_color,
            menu_selector_outline_stroke_thickness,
            menu_option_chosen,
            menu_option_chosen_at,
            menu_option_selection_to_activation_delay,
            al_get_time()
         );
      }

      al_draw_text(
         menu_font,
         this_menu_text_color,
         (int)x,
         (int)(y-h_font_line_height),
         ALLEGRO_ALIGN_CENTER,
         menu_item_text.c_str()
      );

      menu_item_num++;
   }
   return;
}

std::string TitleScreen::infer_current_menu_option_value()
{
   if (menu_options.empty()) return "";
   if (cursor_position < 0 || cursor_position >= menu_options.size())
   {
      throw std::runtime_error("[AllegroFlare/Screens/TitleScreen]: error: cursor_position is not in "
                               "a valid position to get the current menu choice's value.");
   }
   std::string current_menu_option_value = std::get<1>(menu_options[cursor_position]);
   return current_menu_option_value;
}

std::string TitleScreen::infer_current_menu_option_label()
{
   if (menu_options.empty()) return "";
   if (cursor_position < 0 || cursor_position >= menu_options.size())
   {
      throw std::runtime_error("[AllegroFlare/Screens/TitleScreen]: error: cursor_position is not in "
                               "a valid position to get the current menu choice's label.");
   }
   std::string current_menu_option_value = std::get<1>(menu_options[cursor_position]);
   return current_menu_option_value;
}

void TitleScreen::play_menu_move_sound_effect()
{
   if (!(event_emitter))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::play_menu_move_sound_effect]: error: guard \"event_emitter\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::play_menu_move_sound_effect: error: guard \"event_emitter\" not met");
   }
   event_emitter->emit_play_sound_effect_event(menu_move_sound_effect_identifier);
   return;
}

void TitleScreen::play_menu_select_option_sound_effect()
{
   if (!(event_emitter))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::play_menu_select_option_sound_effect]: error: guard \"event_emitter\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::play_menu_select_option_sound_effect: error: guard \"event_emitter\" not met");
   }
   event_emitter->emit_play_sound_effect_event(menu_select_option_sound_effect_identifier);
   return;
}

ALLEGRO_FONT* TitleScreen::obtain_title_font()
{
   if (!(font_bin))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::obtain_title_font]: error: guard \"font_bin\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::obtain_title_font: error: guard \"font_bin\" not met");
   }
   std::stringstream composite_font_str;
   composite_font_str << title_font_name << " " << title_font_size;
   return font_bin->auto_get(composite_font_str.str());
}

ALLEGRO_FONT* TitleScreen::obtain_menu_font()
{
   if (!(font_bin))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::obtain_menu_font]: error: guard \"font_bin\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::obtain_menu_font: error: guard \"font_bin\" not met");
   }
   std::stringstream composite_font_str;
   composite_font_str << menu_font_name << " " << menu_font_size;
   return font_bin->auto_get(composite_font_str.str());
}

ALLEGRO_FONT* TitleScreen::obtain_copyright_font()
{
   if (!(font_bin))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::obtain_copyright_font]: error: guard \"font_bin\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::obtain_copyright_font: error: guard \"font_bin\" not met");
   }
   std::stringstream composite_font_str;
   composite_font_str << copyright_font_name << " " << copyright_font_size;
   return font_bin->auto_get(composite_font_str.str());
}

ALLEGRO_BITMAP* TitleScreen::obtain_title_bitmap()
{
   if (!(bitmap_bin))
   {
      std::stringstream error_message;
      error_message << "[TitleScreen::obtain_title_bitmap]: error: guard \"bitmap_bin\" not met.";
      std::cerr << "\033[1;31m" << error_message.str() << " An exception will be thrown to halt the program.\033[0m" << std::endl;
      throw std::runtime_error("TitleScreen::obtain_title_bitmap: error: guard \"bitmap_bin\" not met");
   }
   return bitmap_bin->auto_get(title_bitmap_name);
}

void TitleScreen::joy_button_down_func(ALLEGRO_EVENT* ev)
{
   // NOTE: These joystick controls are intended to be temporary, and eventually replaced with virtual controls
   // TODO: Replace these with virtual controls
   select_menu_option();
   return;
}

void TitleScreen::joy_axis_func(ALLEGRO_EVENT* ev)
{
   // NOTE: These joystick controls are intended to be temporary, and eventually replaced with virtual controls
   // TODO: Replace these with virtual controls

   static float axis_x = 0;
   static float axis_y = 0;

   int stick = ev->joystick.stick;
   int axis = ev->joystick.axis;
   float pos = ev->joystick.pos;

   float min_stick_break_threshold = 0.4;

   switch (stick)
   {
      case 0: { // The primary joystick, on the left
        if (axis == 0) // horizontal axis
        {
           //if (axis_x < min_stick_break_threshold && pos >= min_stick_break_threshold) 
               //level_select_element.move_cursor_right();
           //if (axis_x > -min_stick_break_threshold && pos <= -min_stick_break_threshold) 
               //level_select_element.move_cursor_left();
           axis_x = pos;
        }
        else if (axis == 1) // vertical axis
        {
           if (axis_y < min_stick_break_threshold && pos >= min_stick_break_threshold) move_cursor_down();
               //level_select_element.move_cursor_down();
           if (axis_y > -min_stick_break_threshold && pos <= -min_stick_break_threshold) move_cursor_up();
               //level_select_element.move_cursor_up();
           axis_y = pos;
        }
      } break;

      case 1: { // The secondary joystick, on the right
      } break;

      case 2: { // The hat, on the left
      } break;
   }

   return;
}

void TitleScreen::virtual_control_button_down_func(AllegroFlare::Player* player, AllegroFlare::VirtualControllers::Base* virtual_controller, int virtual_controller_button_num, bool is_repeat)
{
   if (!processing_user_input_on_main_menu()) return;

   if (virtual_controller_button_num == VirtualControllers::GenericController::BUTTON_UP) move_cursor_up();
   if (virtual_controller_button_num == VirtualControllers::GenericController::BUTTON_DOWN) move_cursor_down();
   if (virtual_controller_button_num == VirtualControllers::GenericController::BUTTON_A
      || virtual_controller_button_num == VirtualControllers::GenericController::BUTTON_MENU
      )
   {
      select_menu_option();
   }
}

bool TitleScreen::menu_is_empty()
{
   return menu_options.empty();
}

bool TitleScreen::menu_has_items()
{
   return !menu_is_empty();
}

std::vector<std::pair<std::string, std::string>> TitleScreen::build_default_menu_options()
{
   std::vector<std::pair<std::string, std::string>> result;
   result = {
      { "Start new game", "start_new_game" },
      { "Credits", "show_credits" },
      { "Exit", "exit_game*" }
   };
   return result;
}

std::vector<std::pair<std::string, std::string>> TitleScreen::build_confirmation_dialog_menu_options()
{
   std::vector<std::pair<std::string, std::string>> result;
   result = {
      { "yes", "exit_game" },
      { "no", "close_confirmation_dialog" },
   };
   return result;
}


} // namespace Screens
} // namespace AllegroFlare


