using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;

public class PlayerHealth : MonoBehaviour
{
    public int maxHealth = 3;
    public int currentHealth;
    public List<Image> hearts;
    public Sprite fullHeart;
    public Sprite emptyHeart;
    public float invincibilityDuration = 2f; // Duration of invincibility frames
    public float flashDuration = 0.1f; // Duration of each flash
    private bool isInvincible = false;

    PlayerMovement playerMovement;
    private SpriteRenderer spriteRenderer; // Reference to the SpriteRenderer component

    [SerializeField] Image heartPrefab;
    public Transform healthContainer;

    private void Start()
    {
        playerMovement = GetComponent<PlayerMovement>();
        spriteRenderer = GetComponent<SpriteRenderer>();
        currentHealth = maxHealth;
        UpdateHealthUI();
    }

    public void TakeDamage(int damage)
    {
        if (isInvincible) return;

        currentHealth -= damage;
        if (currentHealth < 0)
        {
            currentHealth = 0;
        }
        ClearAllHearts();
        UpdateHealthUI();
        if (currentHealth == 0)
        {
            Die();
        }
        else
        {
            StartCoroutine(InvincibilityFrames());
        }
    }

    private void UpdateHealthUI()
    {
        // Add more hearts to the UI
        for (int i = 0; i < maxHealth; i++)
        {
            Image newHeart = Instantiate(heartPrefab, healthContainer);
            hearts.Add(newHeart);
        }

        for (int i = 0; i < maxHealth; i++)
        {
            if (i < currentHealth)
            {
                hearts[i].sprite = fullHeart;
            }
            else
            {
                hearts[i].sprite = emptyHeart;
            }
        }
    }

    private void Die()
    {
        GameManager.instance.GameOver();
    }

    public void IncreaseMaxHealth(int amount)
    {
        maxHealth += amount;
        currentHealth = maxHealth;

        // clear the hearts list
        ClearAllHearts();
        UpdateHealthUI();
    }

    public void Heal(int amount)
    {
        currentHealth += amount;
        if (currentHealth > maxHealth)
        {
            currentHealth = maxHealth;
        }
        ClearAllHearts();
        UpdateHealthUI();
        playerMovement.HealPlayer();
    }

    public void ClearAllHearts()
    {
        foreach (Image heart in hearts)
        {
            Destroy(heart.gameObject);
        }
        hearts.Clear();
    }

    private IEnumerator InvincibilityFrames()
    {
        isInvincible = true;
        float timer = 0f;

        while (timer < invincibilityDuration)
        {
            spriteRenderer.enabled = !spriteRenderer.enabled; // Toggle sprite visibility
            yield return new WaitForSeconds(flashDuration);
            timer += flashDuration;
        }

        spriteRenderer.enabled = true; // Ensure sprite is visible at the end
        isInvincible = false;
    }
}
